use crate::flag::Flag;
use rustc_hash::FxHashMap as HashMap;
use serde::{Deserialize, Serialize};

/// `Node` is a single vertex in the graph-like data structure.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Node {
    /// The index is the node's name, identifying it in the network. It is also *usually* the index
    /// in the `nodes` vector of parent `Network`.
    index: usize,
    /// Map of (link target, link weight) pairs specifying the connections with other nodes in the
    /// parent network.
    links: HashMap<usize, f64>,
    /// The flag hashmap
    pub(crate) flags: HashMap<String, Flag>,
    /// The node degree
    deg: usize,
}

impl Node {
    /// Create a new, empty node
    pub(crate) fn new(index: usize) -> Self {
        Node {
            index,
            links: HashMap::default(),
            flags: HashMap::default(),
            deg: 0,
        }
    }
    /// Get index of the node in it's parent network.
    pub fn index(&self) -> &usize {
        &self.index
    }

    // Link to chosen node
    pub(crate) fn link_to(&mut self, other: usize, value: f64) {
        if self.links.insert(other, value).is_none() {
            self.deg += 1;
        }
    }

    /// Unlink form a chosen node
    pub(crate) fn unlink_from(&mut self, other: usize) {
        if self.links.remove(&other).is_some() {
            self.deg -= 1;
        }
    }

    /// Unlink from all nodes
    pub(crate) fn disconnect(&mut self) {
        self.links.clear()
    }

    /// Get non-mutable reference to the node links.
    pub fn links(&self) -> &HashMap<usize, f64> {
        &self.links
    }

    /// Return non-mutable reference to the node flags.
    pub fn flags(&self) -> &HashMap<String, Flag> {
        &self.flags
    }

    // Get node degree, ie. number of closest neighbors
    pub fn deg(&self) -> &usize {
        &self.deg
    }
}
