//! This module contains the various network constructors
use super::link::Weight;
use super::Model;
use super::Network;

use crate::cn;
use crate::Node;

use core::cell::RefCell;
use rand::distributions::Alphanumeric;
use rand::prelude::*;
use rand_seeder::Seeder;

impl Network {
    /// Create a new network of given size and desired model with a random seed. Use
    /// [`Model::None`] if you want a network with no pre-established connections.
    ///
    /// Note that this is equivalent to calling [`Network::with_seed()`] with a random alphanumeric
    /// seed of length 16.
    ///
    /// # Examples
    /// Creating a network of chosen model and random link weight:
    /// ```
    /// # use cnetworks::*;
    /// let net = Network::new(20, Model::ER { p: 0.4, whole: true }, Weight::Uniform);
    /// assert_eq!(net.size(), &20);
    /// println!("{:?}", net);
    /// ```
    /// Creating a network with no links and establishing them "by hand":
    /// ```
    /// # use cnetworks::*;
    /// # use std::error::Error;
    /// # fn main() -> Result<(), Box<dyn Error>> {
    /// let mut net = Network::new(10, Model::None, Weight::Constant { c: 0.25 });
    /// net.link(1, 2)?;
    /// net.link(4, 7)?;
    /// assert_eq!(net.get_link(1,2)?, Some(0.25));
    /// assert_eq!(net.get_link(4,7)?, Some(0.25));
    /// println!("{:?}", net);
    /// # Ok(())
    /// # }
    /// ```
    pub fn new(size: usize, model: Model, weight: Weight) -> Self {
        Network::with_seed(
            size,
            model,
            weight,
            &thread_rng()
                .sample_iter(&Alphanumeric)
                .take(16)
                .map(char::from)
                .collect::<String>(),
        )
    }

    /// Same as [`Network::new()`], but with a specified `seed` for the internal random number
    /// generator. This makes the network entirely deterministic, allowing full reproducibility
    /// between simulations.
    ///
    /// # Examples
    /// ```
    /// # use cnetworks::*;
    /// let seed = "Boaty McBoatface";
    /// let net1 = Network::with_seed(10, Model::ER { p: 0.75, whole: false, }, Weight::Uniform, seed);
    /// let net2 = Network::with_seed(10, Model::ER { p: 0.75, whole: false, }, Weight::Uniform, seed);
    /// for (node1, node2) in net1.nodes().zip(net2.nodes()) {
    ///     assert_eq!(node1.links(), node2.links());
    /// }
    /// ```
    pub fn with_seed(size: usize, model: Model, weight: Weight, seed: &str) -> Self {
        // Check for value correctness
        if let Weight::Constant { c } = weight {
            if c <= 0. || c > 1. {
                panic!("{}", cn::Err::BadWeight(c));
            }
        }
        // Create the actual network
        let mut net = Network {
            model: Model::None,
            nodes: Vec::with_capacity(size),
            weight,
            size,
            rng: RefCell::new(Seeder::from(seed).make_rng()),
            seed: seed.to_string(),
            indexing_ok: true,
        };
        // Fill in the nodes
        for i in 0..size {
            net.nodes.push(Node::new(i));
        }
        // Initialize links according to the desired model
        match model {
            Model::ER { p, whole } => Model::init_er(&mut net, p, whole),
            Model::BA { m0, m } => Model::init_ba(&mut net, m0, m),
            Model::Custom(_) | Model::None => (),
        }
        net
    }
}

/// Same as [`Network::new()`] with the following properites:
/// - [`Model::None`]
/// - [`Weight::default()`]
/// - [`Network::size()`] of 0
///
/// Note that by default the network is empty. Nodes must be manually attached using
/// [`Network::attach()`].
///
/// # Examples
/// Creating a default network and attaching nodes manually
/// ```
/// # use cnetworks::*;
/// # use std::error::Error;
/// # fn main() -> Result<(), Box<dyn Error>> {
/// let mut net = Network::default();
/// net.attach(&[1, 2])?;
/// net.link(1,2)?;
/// assert_eq!(net.get_link(1,2), Ok(Some(1.0)));
/// # Ok(())
/// # }
/// ```
impl Default for Network {
    fn default() -> Self {
        Network::new(0, Model::None, Weight::default())
    }
}
