/*!
This is a library of tools for creating and manipulating complex networks.

From [Wikipedia](https://en.wikipedia.org/wiki/Complex_network):

> Complex network is a graph-like structure with non-trivial characteristics.
> A complex network is a graph (network) with non-trivial topological features—features that do not
> occur in simple networks such as lattices or random graphs but often occur in networks representing
> real systems. The study of complex networks is a young and active area of scientific
> research inspired largely by empirical findings of real-world networks such
> as computer networks, biological networks, technological networks, brain networks, climate networks
> and social networks.
*/

extern crate rand;
extern crate rand_seeder;
extern crate rand_xoshiro;
extern crate rustc_hash;
extern crate serde;

mod network;

mod algorithms;
mod flag;
mod node;

pub use algorithms::*;
pub use flag::Flag;
pub use network::{Network, Model, Weight};
pub use node::Node;

pub mod cn {
    //! This module contains various library-specific utilities, such as error enums and
    //! auto-implemented iterator traits.

    /// Shorthand trait for an iterator used in this library
    pub trait Iter<I>: DoubleEndedIterator<Item = I> + ExactSizeIterator<Item = I> {}
    impl<T: DoubleEndedIterator + ExactSizeIterator + Iterator<Item = I>, I> Iter<I> for T {}

    /// Error enum used for various recoverable errors encountered while using this library
    #[derive(Debug, PartialEq)]
    pub enum Err {
        NoSuchNode(usize),
        BadWeight(f64),
        LinkToSelf(usize),
        BadProbability(f64),
        NodeExists(usize),
    }

    impl std::fmt::Display for Err {
        fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
            let message = match self {
                Err::NoSuchNode(i) => format!("No such node {}", i),
                Err::BadWeight(w) => format!("Link weight cannot be {}", w),
                Err::LinkToSelf(i) => format!("Attempted linking node {} to itself", i),
                Err::BadProbability(p) => format!("Probability cannot be {}", p),
                Err::NodeExists(i) => format!("Node {} already exists", i),
            };
            write!(f, "{}", message)
        }
    }

    impl std::error::Error for Err {}
}
