extern crate queues;

use queues::*;
use rand::prelude::*;
use rustc_hash::{FxHashMap as HashMap, FxHashSet as HashSet};

use crate::cn;
use crate::flag::Flag;
use crate::Network;

/// Perform a breadth-first search over the network, starting with root and looking for target.
///
/// Returns distance (ie. number of edges) from root to target in the resulting BFS tree or `None`
/// if no path is found. Returns `Err` if the root or target nodes do not exist.
///
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// # use cnetworks::*;
/// # use std::error::Error;
/// # fn main() -> Result<(), Box<dyn Error>> {
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// match bfs(&net, 1, 3)? {
///     Some(d) => println!("Edges between 1 and 3: {}", d),
///     None => println!("No path from 1 to 3 found!"),
/// }
/// # Ok(())
/// # }
/// ```
pub fn bfs(net: &Network, root: usize, target: usize) -> Result<Option<usize>, cn::Err> {
    Ok(bfs_callback(net, root, &[target], |_, _, _| ())?
        .first()
        .unwrap()
        .1)
}

/// Like [`bfs`], but with multiple targets.
///
/// Returns a vector of (target, distance) pairs in the order in which targets were encountered,
/// `Err` if the root or any of the target nodes do not exist.
///
/// A distance of `None` means the target was not enountered, such targets are pushed to the end of
/// the returned vector.
///
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// # use cnetworks::*;
/// # use std::error::Error;
/// # fn main() -> Result<(), Box<dyn Error>> {
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// for (target, distance) in bfs_many( &net, 1, &[2, 3])? {
///     match distance {
///         Some(distance) => println!("Distance from 1 to {} is {}", target, distance),
///         None => println!("Cannot reach {} from 1", target),
///     }
/// }
/// # Ok(())
/// # }
/// ```
pub fn bfs_many(
    net: &Network,
    root: usize,
    targets: &[usize],
) -> Result<Vec<(usize, Option<usize>)>, cn::Err> {
    bfs_callback(net, root, targets, |_, _, _| ())
}

/// Like [`bfs_many()`], but executes `step_callback` on every queue pop.
///
/// The `step_callback` is called for every visited node, providing acesss to current node, it's
/// parent (`None` if it's the root) and a vector of it's children (linked nodes which have not
/// been visited yet).
///
/// Returns`Err` if the root or any of the target nodes do not exist.
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// # use cnetworks::*;
/// # use std::error::Error;
/// # fn main() -> Result<(), Box<dyn Error>> {
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// let mut visited = Vec::new();
/// for (target, distance) in bfs_callback(
///     &net,
///     1,
///     &[2, 3],
///     |current: usize, _parent: Option<usize>, _children: &Vec<usize>| visited.push(current)
/// )?
/// {
///     match distance {
///         Some(d) => println!("Distance from 1 to {} is {}", target, d),
///         None => println!("Cannot reach {} from 1", target),
///     }
/// }
/// println!("Visited {:?} in the process.", visited);
/// # Ok(())
/// # }
/// ```
pub fn bfs_callback<F: FnMut(usize, Option<usize>, &Vec<usize>)>(
    net: &Network,
    root: usize,
    targets: &[usize],
    mut step_callback: F,
) -> Result<Vec<(usize, Option<usize>)>, cn::Err> {
    // Check if the target and root are valid nodes
    for &t in targets {
        net.get(t)?;
    }
    // Looking for these, will remove them as we go
    let mut targets: HashSet<usize> = targets.iter().cloned().collect();
    //
    let mut distances: Vec<(usize, Option<usize>)> = Vec::new();
    let targets_exist = !targets.is_empty();

    net.get(root)?;
    let mut discovered: HashMap<usize, Option<usize>> = HashMap::default();
    // The queue stores (node, node parent) pairs
    let mut to_visit: Queue<(usize, Option<usize>)> = Queue::new();
    // Only the root has a `None` parent
    discovered.insert(root, None);
    to_visit.add((root, None)).expect("Queue error");

    while to_visit.size() != 0 {
        let (current, parent) = to_visit.remove().expect("Queue error");
        let children: Vec<_> = net[current]
            .links()
            .keys()
            .filter(|target| !discovered.contains_key(target))
            .cloned()
            .collect();
        step_callback(current, parent, &children);
        if targets_exist {
            // Remove returns false if value is not in the set
            if targets.remove(&current) {
                // Climb back up the tree until the root is reached
                let mut distance: usize = 0;
                let mut parent = parent;
                loop {
                    match parent {
                        Some(p) => {
                            parent = discovered[&p];
                            distance += 1;
                        }
                        None => {
                            // Record target's distance and continue on
                            distances.push((current, Some(distance)));
                            break;
                        }
                    }
                }
            }
            // Exit condition
            if targets.is_empty() {
                return Ok(distances);
            }
        }
        for i in children {
            discovered.insert(i, Some(current));
            to_visit.add((i, Some(current))).expect("Queue error");
        }
    }
    // No path from root to target was found
    for t in targets {
        distances.push((t, None));
    }
    Ok(distances)
}

/// Build a breadth-first search tree from the network while looking for the specified `targets`.
pub fn bfs_build_tree(net: &Network, root: usize, targets: &[usize]) -> Result<Network, cn::Err> {
    let mut tree = Network::default();
    tree.set_model("BfsTree");
    // It's better to store the nodes here and attach them later
    let mut raw_nodes: Vec<(usize, Vec<usize>)> = Vec::new();
    bfs_callback(net, root, targets, |current, _parent, children| {
        raw_nodes.push((current, children.clone()));
    })?;
    tree.attach(
        &raw_nodes
            .iter()
            .map(|(node, _children)| *node)
            .collect::<Vec<_>>(),
    )?;
    for (node, children) in raw_nodes {
        for child in children {
            tree.link(node, child)?;
        }
    }
    Ok(tree)
}

/// Explore the network starting with `root`, finding all nodes for which the path `root -> node`
/// exists (the **explored**) and those for which it does not (the **unexplored**).
///
/// Returns a tuple of two HashSets: one with the explored nodes (including the root) and another
/// with the unexplored. Returns `Err` if the root does not exist.
///
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// # use cnetworks::*;
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// let (exp, unexp) = explore(&net, 1).unwrap();
/// println!("Available from 1: {:?}", exp);
/// println!("Unavailable from 1: {:?}", unexp);
/// ```
pub fn explore(net: &Network, root: usize) -> Result<(HashSet<usize>, HashSet<usize>), cn::Err> {
    let mut discovered: HashSet<usize> = HashSet::default();
    // Gather the undiscovered nodes
    bfs_callback(net, root, &[], |current, _, _| {
        discovered.insert(current);
    })?;
    let undiscovered: HashSet<usize> = net
        .indexes()
        .filter(|node| !discovered.contains(node))
        .collect();
    Ok((discovered, undiscovered))
}

/// Obtain vector of clusters present in the network, ordered largest to smallest.
///
/// A random element of `net` is chosen as the root and the initial cluster is the first (explored)
/// set returned by `explore(net, root)`. From the second (unexplored) set a new node is chosen as
/// the new root, and the process is repeated until there are no more unexplored nodes.
/// # Examples
/// ```
/// # use cnetworks::*;
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None );
/// let clusters = clusters(&net);
/// println!("Largest cluster: {:?}", clusters.first().expect("No clusters"));
/// println!("Smallest cluster: {:?}", clusters.last().expect("No clusters"));
/// ```
pub fn clusters(net: &Network) -> Vec<HashSet<usize>> {
    let mut clusters: Vec<HashSet<usize>> = Vec::new();
    if net.size() == &0 {
        return clusters;
    }
    // Start from a random node
    let root = net.indexes().choose(&mut *net.rng().unwrap()).unwrap();
    let (expl, mut unexpl) = explore(net, root).unwrap();
    // Add explored to the vector of clusters
    clusters.push(expl);
    loop {
        match unexpl.iter().choose(&mut *net.rng().unwrap()) {
            Some(&new_root) => {
                // Establish its cluster
                let (cls, _) = explore(net, new_root).unwrap();
                // Mark everything in the cluster as explored
                for c in &cls {
                    unexpl.remove(c);
                }
                clusters.push(cls);
            }
            // The are no unexplored nodes
            None => {
                // Order cluster largest -> smallest
                clusters.sort_by_key(|set| -(set.len() as i64));
                return clusters;
            }
        }
    }
}

/// Stitch the network together if it is not whole.
///
/// Connects random elements of smaller clusters to the largest cluster, extending it until it
/// covers the entire network, making the network "whole".
pub fn stitch_together(net: &mut Network) {
    // Get the clusters
    let clusters = clusters(net);
    let mut largest = clusters.first().unwrap().clone();
    for cluster in &clusters[1..] {
        // Connect random node from current cluster to random node from the largest cluster
        let &current = cluster.iter().choose(&mut *net.rng().unwrap()).unwrap();
        let &other = largest.iter().choose(&mut *net.rng().unwrap()).unwrap();
        net.link(current, other).unwrap();
        // Extend the largest cluster with the just-connected nodes
        largest.extend(cluster.iter());
    }
}

/// Simulate spread of information according to the **susceptible-infected** diffusion model.
///
/// On each time step the nodes for which the `infected` flag is `true` will try to infect
/// their closest neighbors with probability given by link weight, setting their
/// "INFECTED" flag.
///
/// Returns the root of the infection.
pub fn spread_si(net: &mut Network, steps: usize) -> usize {
    // Choose random node as origin
    let root = net.indexes().choose(&mut *net.rng().unwrap()).unwrap();
    net.flag(root, "INFECTED", Flag::None).unwrap();
    for _ in 0..steps {
        let mut to_infect = Vec::new();
        // Find nodes to infect
        for (index, _) in net.flagged("INFECTED") {
            // Try to infect the neighbors
            for (target, weight) in net[index].links() {
                if net.rng().unwrap().gen::<f64>() < *weight {
                    // Target is getting infected
                    to_infect.push(*target);
                }
            }
        }
        // Infect nodes marked for infection
        for target in to_infect {
            net.flag(target, "INFECTED", Flag::None).unwrap();
        }
    }
    root
}
