//! This module contains public interface for obtaining network properties.

use super::link::Weight;
use super::Model;
use super::Network;

use crate::CnErr;
use crate::Node;

use core::cell::{BorrowMutError, RefMut};
use rand::prelude::*;
use rustc_hash::FxHashMap as HashMap;

impl Network {
    /// Return the network size, ie. the total number of nodes.
    pub fn size(&self) -> &usize {
        &self.size
    }

    /// Return the model of the network
    pub fn model(&self) -> &Model {
        &self.model
    }

    /// Return the type of link weight
    pub fn weight(&self) -> &Weight {
        &self.weight
    }

    /// Return the network-local rng thread seed.
    ///
    /// This is a random alphanumeric string of length 16, unless explicitly specified in
    /// [`Network::with_seed()`].
    pub fn seed(&self) -> &str {
        &self.seed
    }

    /// Try to borrow the network-local rng thread from it's `RefCell`. This is effectively
    /// calling `RefCell::try_borrow_mut()`.
    pub fn rng(&self) -> Result<RefMut<'_, impl RngCore>, BorrowMutError> {
        self.rng.try_borrow_mut()
    }

    /// Return iterator containing the (index, node) pairs.
    ///
    /// Note that this is **not** strictly equivalent to `Network::nodes().enumerate()`, since a
    /// node's index **does not**  always equal it's position in the iterator, even though nodes
    /// are always ordered by their index.
    ///
    /// For example it is possible to have a network with indexes (0, 2, 3) or (100, 101,
    /// 150), but not (10, 0, 1).
    pub fn iter(
        &self,
    ) -> impl DoubleEndedIterator<Item = (usize, &Node)> + ExactSizeIterator<Item = (usize, &Node)> + '_
    {
        self.nodes.iter().map(|node| (*node.index(), node))
    }


    /// Return iterator containing the node indexes.
    ///
    /// Use [`Network::nodes()`] to iterate over the nodes themselves and [`Network::iter()`] to use both indexes
    /// and values at once.
    ///
    /// Keep in mind that nodes are **always** ordered by their index, but it's possible to have a
    /// network of nodes with non-sequential indexes, such as (0,2,3) or (100, 101, 150), but not
    /// (10, 0 1).
    pub fn indexes(
        &self,
    ) -> impl DoubleEndedIterator<Item = usize> + ExactSizeIterator<Item = usize> + '_ {
        self.nodes.iter().map(|node| *node.index())
    }

    /// Return iterator containing the network nodes, ordered by their index.
    pub fn nodes(
        &self,
    ) -> impl DoubleEndedIterator<Item = &Node> + ExactSizeIterator<Item = &Node> {
        self.nodes.iter()
    }

    /// Return immutable reference to a given node if it exists, `Err` if it does not.
    pub fn get(&self, index: usize) -> Result<&Node, CnErr> {
        if self.indexing_ok {
            // The node can be found at the specified index
            match self.nodes.get(index) {
                Some(node) => Ok(node),
                None => Err(CnErr::NoSuchNode(index)),
            }
        } else {
            // Binary search is required
            match self
                .nodes
                .binary_search_by_key(&index, |node| *node.index())
            {
                Ok(true_idx) => Ok(&self.nodes[true_idx]),
                Err(..) => Err(CnErr::NoSuchNode(index)),
            }
        }
    }

    /// Return mutable reference to a given node if it exists, `Err` if it does not.
    pub(crate) fn get_mut(&mut self, index: usize) -> Result<&mut Node, CnErr> {
        if self.indexing_ok {
            // The node can be found at the specified index
            match self.nodes.get_mut(index) {
                Some(node) => Ok(node),
                None => Err(CnErr::NoSuchNode(index)),
            }
        } else {
            // Binary search is required
            match self
                .nodes
                .binary_search_by_key(&index, |node| *node.index())
            {
                Ok(true_idx) => Ok(&mut self.nodes[true_idx]),
                Err(..) => Err(CnErr::NoSuchNode(index)),
            }
        }
    }

    /// Return the number of connected nodes in the network, ie. ones that have at least one link
    pub fn size_connected(&self) -> usize {
        self.nodes()
            .filter(|&node| !node.links().is_empty())
            .count()
    }

    /// Return the total number of edges in the network.
    pub fn edges(&self) -> usize {
        self.total_deg() / 2
    }

    /// Calculate the arithmetical average of vertex degrees (ie. number of closest neighbors)
    /// in the network.
    pub fn avg_deg(&self) -> f64 {
        self.total_deg() as f64 / self.size as f64
    }

    /// Get the total degree of the network, ie. sum of degrees over all of the nodes.
    pub fn total_deg(&self) -> usize {
        self.nodes().map(|node| node.deg()).sum()
    }

    /// Return the degree distribution of the network as a `HashMap` of (degree, occurrences) pairs.
    pub fn deg_distr(&self) -> HashMap<usize, usize> {
        let mut bins: HashMap<usize, usize> = HashMap::default();
        for node in self.nodes() {
            let deg = node.deg();
            match bins.get_key_value(deg) {
                // If there already is such bin then increment it by 1
                Some((&key, &value)) => bins.insert(key, value + 1),
                // If there isn't create one with single occurrence
                None => bins.insert(*deg, 1),
            };
        }
        bins
    }
}
