//! This module contains methods for `Network` related to creating and removing links as well as
//! disconnecting entire nodes from the network.

use super::models::Model;
use super::Network;
use crate::algorithms::bfs;
use crate::error::CnErr;
use crate::Node;
use rand::prelude::*;
use rustc_hash::FxHashMap as HashMap;
use serde::{Deserialize, Serialize};

/// `Weight` determines the weight of each link in the network (ie. the probability of the
/// infection spreading through it in each time step).
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum Weight {
    /// The probability should be constant end equalt to `c`. Keep in mind that 0 < c <= 1, the
    /// `Network::new()` constructor **will panic** if that is not the case.
    Constant { c: f64 },
    /// A weight sampled uniformly from (0, 1).
    Uniform,
    /// For when the link weight does not matter, treated like Weight::Constant { c: 1.0 }
    None,
}

impl Default for Weight {
    fn default() -> Self {
        Self::None
    }
}

impl Network {
    /// Get the weight of the link between nodes `i` and `j`.
    ///
    /// Returns `None` if such link does not exist and `Err` if `i` and `j` are the same or do not
    /// exist.
    pub fn get_link(&self, i: usize, j: usize) -> Result<Option<f64>, CnErr> {
        let node_i = self.get(i)?;
        self.get(j)?;
        if i == j {
            return Err(CnErr::LinkToSelf(i));
        }
        Ok(node_i.links().get(&j).cloned())
    }

    /// Establish a link between nodes `i` and `j` using current network's weight.
    ///
    /// Returns the weight if the requested link already exist, `None` if it does not. Returns
    /// `Err` if `i` and `j` are the same or do not exist.
    pub fn link(&mut self, i: usize, j: usize) -> Result<Option<f64>, CnErr> {
        let w = match self.weight {
            Weight::Constant { c } => c,
            Weight::Uniform => self.rng().unwrap().gen(),
            Weight::None => 1.0,
        };
        self.link_exact(i, j, w)
    }

    /// Establish a link between nodes i and j with specified `weight`.
    ///
    /// Updates the link weight and returns the old value if it already exist, `None` otherwise.
    /// Returns `Err` if `i` and `j` are the same or do not exist, or if the `weight` is not in the range
    /// (0, 1].
    pub fn link_exact(&mut self, i: usize, j: usize, weight: f64) -> Result<Option<f64>, CnErr> {
        let ret = self.get_link(i, j)?;
        if weight <= 0. || weight > 1. {
            return Err(CnErr::BadWeight(weight));
        }
        if i == j {
            return Err(CnErr::LinkToSelf(i));
        }
        self.get_mut(i).unwrap().link_to(j, weight);
        self.get_mut(j).unwrap().link_to(i, weight);
        Ok(ret)
    }

    /// Unsafely remove a link between two nodes. Does not perform a check whether the integrity of
    /// the network is preserved.
    ///
    /// Returns the weight of the removed connection or `None` if the connection does not exist and
    /// `Err` if `i` and `j` are the same or do not exist.
    pub fn unlink(&mut self, i: usize, j: usize) -> Result<Option<f64>, CnErr> {
        match self.get_link(i, j)? {
            Some(weight) => {
                self.get_mut(i).unwrap().unlink_from(j);
                self.get_mut(j).unwrap().unlink_from(i);
                Ok(Some(weight))
            }
            None => Ok(None),
        }
    }

    /// Safely remove a link between two nodes ensuring the network does not split. Performs a BFS
    /// search to check whether the integrity of the network is preserved.
    ///
    /// Returns the weight of removed link or `None` if the link does not exist or cannot be safely
    /// removed. Returns `Err` if i and j are the same or do not exist.
    pub fn unlink_safe(&mut self, i: usize, j: usize) -> Result<Option<f64>, CnErr> {
        let w = match self.unlink(i, j)? {
            Some(weight) => weight,
            None => return Ok(None),
        };
        match bfs(self, i, j)? {
            Some(..) => Ok(Some(w)),
            None => {
                // There is no longer a path from i to j - revert the unlink
                self.link_exact(i, j, w)?;
                Ok(None)
            }
        }
    }

    /// Unsafely disconnect a node from the network by removing all of its links. Does not
    /// check if the network integrity is perserved.
    ///
    /// Returns the removed links as a `HashMap` of (target, weight) pairsm `Err` if `i`
    /// and `j` are the same or do not exist.
    pub fn disconnect(&mut self, node: usize) -> Result<HashMap<usize, f64>, CnErr> {
        let node = self.get_mut(node)?;
        let old_links = node.links().clone();
        node.disconnect();
        let node = *node.index();
        for &idx in old_links.keys() {
            self.get_mut(idx).unwrap().unlink_from(node);
        }
        Ok(old_links)
    }

    /// Safely disconnect a node from the network by removing all of its links. Performs a bfs
    /// search to ensure that the other nodes can stil all reach each other, ie. the integrity of
    /// the network is perserved.
    ///
    /// Returns removed links as `HashMap` of (target, weight) pairs or `None` if `node` cannot be
    /// safely disconnected.
    ///
    /// Returns `Err` if `node` does not exist.
    pub fn disconnect_safe(&mut self, node: usize) -> Result<Option<HashMap<usize, f64>>, CnErr> {
        // Blindly disconnect the requested node
        let links = self.disconnect(node)?;
        // Get a vector of used-to-be-connected nodes
        let used_to: Vec<&usize> = links.keys().collect();
        // i and j are indexes in the keys vector and the values of keys[i] and keys[j] are
        // the used-to-be-connected nodes in self.nodes
        for (i, node_i) in used_to.iter().enumerate() {
            // For every node that used to connect to the removed...
            for node_j in &used_to[i..] {
                // ... for all nodes after it in the used_to vector...
                // ... check if path between them still exists
                if bfs(self, **node_i, **node_j)?.is_none() {
                    // There is no longer a path from node_i to node_j - revert the disconnect
                    for (key, value) in links {
                        self.link_exact(key, node, value)?;
                    }
                    return Ok(None);
                }
            }
        }
        Ok(Some(links))
    }

    /// Removes **ALL** links in the network and sets the `model` to `Model::None` such that
    /// the initial linking process can be conducted again, eg. using a different model in
    /// combination with `Network::init_*`.
    pub fn disconnect_all(&mut self) {
        for node in &mut self.nodes {
            node.disconnect();
        }
        self.model = Model::None;
    }

    /// Attach new nodes with specified indexes. Note that the nodes must be connected to the rest
    /// of the network manually, as no links will be automatically established.
    ///
    /// Returns `Err` if there already exists a node with any of the specified indexes. No nodes
    /// will be attached in that case.
    pub fn attach(&mut self, indexes: &[usize]) -> Result<(), CnErr> {
        // Check for existance first
        for &i in indexes {
            if self.get(i).is_ok() {
                return Err(CnErr::NodeExists(i));
            }
        }
        // Only attach if all idexes in slice are unoccupied.
        for &i in indexes {
            self.nodes.push(Node::new(i));
            self.size += 1;
        }
        self.nodes.sort_unstable_by_key(|node| *node.index());
        let mut indexing_ok = self.indexing_ok;
        if indexing_ok {
            // Reverse the iter since beggining nodes were likely added during network creation
            for (true_index, node) in self.nodes().enumerate().rev() {
                if node.index() != &true_index {
                    indexing_ok = false;
                    break;
                }
            }
        }
        self.indexing_ok = indexing_ok;
        Ok(())
    }

    /// Completely remove the node with specified `index` from the network, disconnecting it first.
    ///
    /// Returns `Err` if node with specified index does not exist.
    pub fn detach(&mut self, index: usize) -> Result<(), CnErr> {
        self.disconnect(index)?;
        if self.indexing_ok {
            self.nodes.remove(index);
        } else {
            let true_index = self
                .nodes
                .binary_search_by_key(&index, |node| *node.index())
                .unwrap(); // We can unwrap since the disconnect already checks for existance
            self.nodes.remove(true_index);
        }
        self.size -= 1;
        self.indexing_ok = false;
        Ok(())
    }
}
