//! This module contains the error enum.

use std::error::Error;
use std::fmt::{Display, Formatter, Result as FmtResult};

/// `Error` is used to signify a recoverable error.
#[derive(Debug, PartialEq)]
pub enum CnErr {
    NoSuchNode(usize),
    BadWeight(f64),
    LinkToSelf(usize),
    BadProbability(f64),
    NodeExists(usize),
}

impl Error for CnErr {}

impl Display for CnErr {
    /// Used to panic and display a meaningful message when the error is non-recoverable.
    fn fmt(&self, f: &mut  Formatter) -> FmtResult {
        let message = match self {
            CnErr::NoSuchNode(i) => format!("No such node {}", i),
            CnErr::BadWeight(w) => format!("Link weight cannot be {}", w),
            CnErr::LinkToSelf(i) => format!("Attempted linking node {} to itself", i),
            CnErr::BadProbability(p) => format!("Probability cannot be {}", p),
            CnErr::NodeExists(i) => format!("Node {} already exists", i),
        };
        write!(f, "{}", message)
    }
}
