//! Test linking/unlinking and connecting/disconnecting nodes.
use cnetworks::*;
use rustc_hash::FxHashSet as HashSet;

#[test]
fn test_bfs() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(
        bfs(&net, 0, 2).unwrap(),
        Some(2),
    );
    net.unlink(0, 1).unwrap();
    assert_eq!(bfs(&net, 0, 2).unwrap(), None);
}

#[test]
fn test_bfs_callback() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    let mut visited: Vec<usize> = vec![];
    assert_eq!(
        bfs_callback(&net, 0, 2, |current, _, _| visited.push(current)).unwrap(),
        Some(2),
    );
    assert_eq!(visited, vec![0,1,2]);
    net.unlink(0, 1).unwrap();
    assert_eq!(bfs(&net, 0, 2).unwrap(), None);
}

#[test]
fn test_explore() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    for node in [0, 1, 2] {
        assert_eq!(
            explore(&net, node).unwrap(),
            (vec![0, 1, 2].into_iter().collect(), HashSet::default())
        );
    }
    net.unlink(1, 2).unwrap();
    for node in [0, 1] {
        assert_eq!(
            explore(&net, node).unwrap(),
            (
                vec![0, 1].into_iter().collect(),
                vec![2].into_iter().collect()
            )
        );
    }
    assert_eq!(
        explore(&net, 2).unwrap(),
        (
            vec![2].into_iter().collect(),
            vec![1, 0].into_iter().collect()
        )
    );
}

#[test]
fn test_clusters() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(
        clusters(&net),
        vec![
            vec![0, 1, 2].into_iter().collect(),
            vec![3].into_iter().collect()
        ]
    );
    net.link(1, 3).unwrap();
    let clst = clusters(&net);
    assert_eq!(clst, vec![vec![0, 1, 2, 3].into_iter().collect()]);
}

#[test]
fn test_spread_si() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    net.link(2, 3).unwrap();
    let root = spread_si(&mut net, 1);
    let mut infected = net
        .flagged("INFECTED")
        .keys()
        .cloned()
        .collect::<Vec<usize>>();
    infected.sort_unstable();
    match root {
        0 => assert_eq!(infected, vec![0, 1]),
        1 => assert_eq!(infected, vec![0, 1, 2]),
        2 => assert_eq!(infected, vec![1, 2, 3]),
        3 => assert_eq!(infected, vec![2, 3]),
        _ => panic!("Root should not be {}", root),
    }
    net.clear_flag("INFECTED");
    let root = spread_si(&mut net, 3);
    assert!(vec![0, 1, 2, 3].contains(&root));
    let mut infected = net
        .flagged("INFECTED")
        .keys()
        .cloned()
        .collect::<Vec<usize>>();
    infected.sort_unstable();
    assert_eq!(infected, vec![0, 1, 2, 3])
}

#[test]
fn test_stitch() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    stitch_together(&mut net);
    for start in [0, 1, 2, 3] {
        let (_, unexpl) = explore(&net, start).unwrap();
        assert!(unexpl.is_empty());
    }
}
