//! This module contains public interface for obtaining network properties.

use super::link::Weight;
use super::Model;
use super::Network;

use crate::CNErr;
use crate::Node;

use core::cell::{BorrowMutError, RefMut};
use rand::prelude::*;
use rustc_hash::FxHashMap as HashMap;

impl Network {
    /// Return the network size, ie. the total number of nodes.
    pub fn size(&self) -> usize {
        self.size
    }

    /// Return the model of the network
    pub fn model(&self) -> &Model {
        &self.model
    }

    /// Return the type of link weight
    pub fn weight(&self) -> &Weight {
        &self.weight
    }

    /// Try to borrow the network-local rng thread from it's `RefCell`. This is effectively to
    /// calling a `RefCell::try_borrow_mut()`.
    pub fn rng(&self) -> Result<RefMut<'_, impl RngCore>, BorrowMutError> {
        self.rng.try_borrow_mut()
    }

    /// Return iterator containing the node indexes. Use `nodes()` to iterate over the nodes
    /// themselves and `enumerated()` to use both indexes and values at once.
    pub fn indexes(&self) -> impl Iterator<Item = usize> + '_ {
        self.nodes.iter().map(|node| *node.index())
    }

    /// Return iterator containing the network nodes.
    pub fn nodes(&self) -> impl Iterator<Item = &Node> {
        self.nodes.iter()
    }

    /// Return iterator containing the (index, node) pairs.
    pub fn enumerated(&self) -> impl Iterator<Item = (usize, &Node)> + '_ {
        self.nodes.iter().map(|node| (*node.index(), node))
    }

    /// Returns immutable reference to a given node if it exists, `Err` if it does not.
    pub fn get(&self, index: usize) -> Result<&Node, CNErr> {
        if self.indexing_ok {
            // The node can be found at the specified index
            match self.nodes.get(index) {
                Some(node) => Ok(node),
                None => Err(CNErr::NoSuchNode(index)),
            }
        } else {
            // Binary search is required
            match self
                .nodes
                .binary_search_by_key(&index, |node| *node.index())
            {
                Ok(true_idx) => Ok(&self.nodes[true_idx]),
                Err(..) => Err(CNErr::NoSuchNode(index)),
            }
        }
    }

    /// Returns mutable reference to a given node if it exists, `Err` if it does not.
    pub(crate) fn get_mut(&mut self, index: usize) -> Result<&mut Node, CNErr> {
        if self.indexing_ok {
            // The node can be found at the specified index
            match self.nodes.get_mut(index) {
                Some(node) => Ok(node),
                None => Err(CNErr::NoSuchNode(index)),
            }
        } else {
            // Binary search is required
            match self
                .nodes
                .binary_search_by_key(&index, |node| *node.index())
            {
                Ok(true_idx) => Ok(&mut self.nodes[true_idx]),
                Err(..) => Err(CNErr::NoSuchNode(index)),
            }
        }
    }

    /// Return the number of connected nodes in the network, ie. ones that have at least one link
    pub fn size_connected(&self) -> usize {
        self.nodes()
            .filter(|&node| !node.links().is_empty())
            .count()
    }

    /// Return the total number of edges in the network.
    pub fn edges(&self) -> usize {
        self.total_deg() / 2
    }

    /// Calculate the arithmetical average of vertex degrees (ie. number of closest neighbors)
    /// in the network.
    pub fn avg_deg(&self) -> f64 {
        self.total_deg() as f64 / self.size as f64
    }

    /// Get the total degree of the network, ie. sum of degrees over all of the nodes.
    pub fn total_deg(&self) -> usize {
        self.nodes().map(|node| node.deg()).sum()
    }

    /// Return the degree distribution of the network as a `HashMap` of (degree, occurrences) pairs.
    pub fn deg_distr(&self) -> HashMap<usize, usize> {
        let mut bins: HashMap<usize, usize> = HashMap::default();
        for node in self.nodes() {
            let deg = node.deg();
            match bins.get_key_value(deg) {
                // If there already is such bin then increment it by 1
                Some((&key, &value)) => bins.insert(key, value + 1),
                // If there isn't create one with single occurrence
                None => bins.insert(*deg, 1),
            };
        }
        bins
    }
}
