//! This module contains the network constructors
use super::link::Weight;
use super::Model;
use super::Network;

use crate::CNErr;
use crate::Node;

use core::cell::RefCell;
use rand_xoshiro::rand_core::SeedableRng;
use rand_xoshiro::Xoshiro256Plus;

impl Network {
    /// Create a new network of given size and desired model. Use `Model::None` if you want
    /// a network with no connections.
    /// # Examples
    /// Creating a network of chosen model and random link weight:
    /// ```
    /// use cnetworks::*;
    /// let net = Network::new(20, Model::ER { p: 0.4, whole: true }, Weight::Uniform);
    /// println!("{:?}", net);
    /// ```
    /// Creating a network with no links and establishing them "by hand":
    /// ```
    /// use cnetworks::*;
    /// let mut net = Network::new(10, Model::None, Weight::Constant { c: 0.25 });
    /// net.link(1, 2);
    /// net.link(4, 7);
    /// println!("{:?}", net);
    /// ```
    pub fn new(size: usize, model: Model, weight: Weight) -> Self {
        // Check for value correctness
        if size == 0 {
            panic!("Attempted creation of network with no nodes.");
        }
        if let Weight::Constant { c } = weight {
            if c <= 0. || c > 1. {
                CNErr::BadWeight(c).panic();
            }
        }
        // Create the network
        let mut net = Network {
            model: Model::None,
            nodes: Vec::with_capacity(size),
            weight,
            size,
            rng: RefCell::new(Xoshiro256Plus::from_entropy()),
            indexing_ok: true,
        };
        // Fill in the nodes
        for i in 0..size {
            net.nodes.push(Node::new(i));
        }
        // Initialize links according to the desired model
        match model {
            Model::ER { p, whole } => Model::init_er(&mut net, p, whole),
            Model::BA { m0, m } => Model::init_ba(&mut net, m0, m),
            Model::None => (),
        }
        net
    }
}
