//! This module contains methods for `Network` related to creating and removing links as well as
//! disconnecting entire nodes from the network.

use crate::algorithms::bfs;
use crate::error::CNErr;
use super::models::Model;
use super::Network;
use rand::prelude::*;
use rustc_hash::FxHashMap as HashMap;
use serde::{Serialize, Deserialize};

/// `Weight` determines the weight of each link in the network (ie. the probability of the
/// infection spreading through it in each time step).
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum Weight {
    /// The probability should be constant end equalt to `c`. Keep in mind that 0 < c <= 1, the
    /// `Network::new()` constructor **will panic** if that is not the case.
    Constant { c: f64 },
    /// A weight sampled uniformly from (0, 1).
    Uniform,
    /// For when the link weight does not matter, treated like Weight::Constant { c: 1.0 }
    None,
}

impl Default for Weight {
    fn default() -> Self {
        Self::None
    }
}

impl Network {
    /// Get the weight of the link between nodes i and j.
    ///
    /// Returns `None` if such link does not exist and `Err` if i and j are the same or do not
    /// exist.
    pub fn get_link(&self, i: usize, j: usize) -> Result<Option<f64>, CNErr> {
        let node_i = self.get(i)?;
        self.get(j)?;
        if i == j {
            return Err(CNErr::LinkToSelf(i));
        }
        Ok(node_i.links().get(&j).cloned())
    }

    /// Establish a link between nodes i and j with current network's weight.  Returns the weight
    /// if the requested link already exist, `None` if it does not. Returns `Err` if i
    /// and j are the same or do not exist.
    pub fn link(&mut self, i: usize, j: usize) -> Result<Option<f64>, CNErr> {
        let w = match self.weight {
            Weight::Constant { c } => c,
            Weight::Uniform => self.rng().unwrap().gen(),
            Weight::None => 1.0,
        };
        self.link_exact(i, j, w)
    }

    /// Establish a link between nodes i and j with specified `weight`. Updates the link weight and
    /// returns the old value if it already exist, `None` otherwise. Returns `Err` if i
    /// and j are the same or do not exist, or if the weight is not in the range (0, 1].
    pub fn link_exact(&mut self, i: usize, j: usize, w: f64) -> Result<Option<f64>, CNErr> {
        let ret = self.get_link(i, j)?;
        if w <= 0. || w > 1. {
            return Err(CNErr::BadWeight(w));
        }
        if i == j {
            return Err(CNErr::LinkToSelf(i));
        }
        self.get_mut(i).unwrap().link_to(j, w);
        self.get_mut(j).unwrap().link_to(i, w);
        Ok(ret)
    }

    /// Unsafely remove a link between two nodes. Does not perform a check whether the integrity of
    /// the network is preserved.
    ///
    /// Returns the weight of the removed connection or `None` if the connection does not exist and
    /// `Err` if i and j are the same or do not exist.
    pub fn unlink(&mut self, i: usize, j: usize) -> Result<Option<f64>, CNErr> {
        match self.get_link(i, j)? {
            Some(weight) => {
                self.get_mut(i).unwrap().unlink_from(j);
                self.get_mut(j).unwrap().unlink_from(i);
                Ok(Some(weight))
            }
            None => Ok(None),
        }
    }

    /// Safely remove a link between two nodes ensuring the network does not split. Performs a BFS
    /// search to check whether the integrity of the network is preserved.
    ///
    /// Returns the weight of removed link or `None` if the link does not exist or cannot be safely
    /// removed. Returns `Err` if i and j are the same or do not exist.
    pub fn unlink_safe(&mut self, i: usize, j: usize) -> Result<Option<f64>, CNErr> {
        let w = match self.unlink(i, j)? {
            Some(weight) => weight,
            None => return Ok(None),
        };
        match bfs(self, i, j)? {
            Some(..) => Ok(Some(w)),
            None => {
                // There is no longer a path from i to j - revert the unlink
                self.link_exact(i, j, w)?;
                Ok(None)
            }
        }
    }

    /// Unsafely disconnect a node from the network by removing all of its links. Does not
    /// check if the network integrity is perserved.
    ///
    /// Returns the removed links as a `HashMap` of (target, weight) pairs. Returns `Err` if i
    /// and j are the same or do not exist.
    pub fn disconnect(&mut self, node: usize) -> Result<HashMap<usize, f64>, CNErr> {
        let node = self.get_mut(node)?;
        let old_links = node.links().clone();
        node.disconnect();
        let node = *node.index();
        for &idx in old_links.keys() {
            self.get_mut(idx).unwrap().unlink_from(node);
        }
        Ok(old_links)
    }

    /// Safely disconnect a node from the network by removing all of its links. Performs a bfs
    /// search to ensure that the other nodes can stil all reach each other, ie. the integrity of
    /// the network is perserved.
    ///
    /// Returns removed links as `HashMap` of (target, weight) pairs or `None` if `node` cannot be
    /// safely disconnected. Returns `Err` if `node` does not exist.
    pub fn disconnect_safe(&mut self, node: usize) -> Result<Option<HashMap<usize, f64>>, CNErr> {
        // Blindly disconnect the requested node
        let links = self.disconnect(node)?;
        // Get a vector of used-to-be-connected nodes
        let used_to: Vec<&usize> = links.keys().collect();
        // i and j are indexes in the keys vector and the values of keys[i] and keys[j] are
        // the used-to-be-connected nodes in self.nodes
        for (i, node_i) in used_to.iter().enumerate() {
            // For every node that used to connect to the removed...
            for node_j in &used_to[i..] {
                // ... for all nodes after it in the used_to vector...
                // ... check if path between them still exists
                if bfs(self, **node_i, **node_j)?.is_none() {
                    // There is no longer a path from node_i to node_j - revert the disconnect
                    for (key, value) in links {
                        self.link_exact(key, node, value)?;
                    }
                    return Ok(None);
                }
            }
        }
        Ok(Some(links))
    }

    /// Removes **ALL** links in the network and sets the `model` to `Model::None` such that
    /// the initial linking process can be conducted again, eg. using a different model in
    /// combination with `Network::init_*`.
    pub fn disconnect_all(&mut self) {
        for node in &mut self.nodes {
            node.disconnect();
        }
        self.model = Model::None;
    }
}
