//! This module contains the `Flag` enum, used to store flags of a particular node
use super::Network;
use crate::error::CNErr;
use rustc_hash::FxHashMap as HashMap;
use serde::{Serialize, Deserialize};

/// `Flag` is used to store specific information about a given node.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum FlagContent {
    /// String flag
    Str(String),
    /// Unsigned integer flag
    Uint(usize),
    /// Signed integer flag
    Int(isize),
    /// Floating point number flag
    Float(f64),
    /// A flag with no contents
    None,
}

impl Network {
    /// Set a flag with specified name and content. Updates existing flag and returns the old
    /// contents, or `None` if the flag was not set. Returns `Err` if specified node does not
    /// exist.
    pub fn flag(
        &mut self,
        index: usize,
        name: &str,
        content: FlagContent,
    ) -> Result<Option<FlagContent>, CNErr> {
        Ok(self.get_mut(index)?.flags.insert(name.to_string(), content))
    }

    /// Remove a flag with specified name. Returns the contents or `None` if flag was not set.
    /// Returns `Err` if specified node does not exist.
    pub fn unflag(&mut self, index: usize, name: &str) -> Result<Option<FlagContent>, CNErr> {
        Ok(self.get_mut(index)?.flags.remove(name))
    }

    /// Get all nodes containing given flag. Returns HashMap of (node index, flag value) pairs.
    pub fn flagged(&self, name: &str) -> HashMap<usize, FlagContent> {
        self.nodes()
            .filter(move |node| node.flags().contains_key(name))
            .map(move |node| (*node.index(), node.flags()[name].clone()))
            .collect::<HashMap<usize, FlagContent>>()
    }

    /// Get all nodes not containing given flag. Returns vector of node indexes.
    pub fn not_flagged(&self, name: &str) -> Vec<usize> {
        self.nodes()
            .filter(move |node| !node.flags().contains_key(name))
            .map(move |node| *node.index())
            .collect::<Vec<usize>>()
    }

    /// Clear the flag from all nodes. Returns iterator of the removed (node index, flag value)
    /// pairs.
    pub fn clear_flag(&mut self, name: &str) {
        for (index, _) in self.flagged(name) {
            self.unflag(index, name).unwrap();
        }
    }
}
