//! This module contains the main `Network` graph-like data structure.
mod link;
mod models;
mod new;
mod props;

pub use link::*;
pub use models::Model;
pub use new::*;
pub use props::*;

use crate::Node;
use rand_xoshiro::Xoshiro256StarStar;
use rustc_hash::FxHashMap as HashMap;
use serde::{Deserialize, Serialize};
use std::cell::RefCell;

/// `Network` is the main graph-like data structure.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Network {
    /// The main data structure.
    nodes: HashMap<usize, Node>,
    /// The model used to initialize links between nodes.
    model: Model,
    /// The distribution followed by the link weights.
    weight: Weight,
    /// The network's randomness source
    rng: RefCell<Xoshiro256StarStar>,
    /// The network size (total number of nodes in the network)
    size: usize,
    /// The network integrity, ie. if nodes[i].index() == i.
    integrity_kept: bool,
}

/// Enables easy non-mutable access to the nodes using the indexing operator.
impl std::ops::Index<usize> for Network {
    type Output = Node;
    fn index(&self, idx: usize) -> &Self::Output {
        &self.nodes[&idx]
    }
}
