extern crate queues;

use queues::*;
use rand::prelude::*;
use rustc_hash::{FxHashMap as HashMap, FxHashSet as HashSet};

use crate::error::CNErr;
use crate::FlagContent;
use crate::Network;

/// Obtain vector of clusters present in the network, ordered largest to smallest.
///
/// A random element of `net` is chosen as the root and the initial cluster is the first (explored)
/// set returned by `explore(net, root)`. From the second (unexplored) set a new node is chosen as
/// the new root, and the process is repeated until there are no more unexplored nodes.
/// # Examples
/// ```
/// use cnetworks::*;
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None );
/// let clusters = clusters(&net);
/// println!("Largest cluster: {:?}", clusters.first().unwrap());
/// println!("Smallest cluster: {:?}", clusters.last().unwrap());
/// ```
pub fn clusters(net: &Network) -> Vec<HashSet<usize>> {
    let mut clusters: Vec<HashSet<usize>> = Vec::new();
    // Start from a random node
    let root = net.indexes().choose(&mut *net.rng().unwrap()).unwrap();
    let (expl, mut unexpl) = explore(net, root).unwrap();
    // Add explored to the vector of clusters
    clusters.push(expl);
    loop {
        match unexpl.iter().choose(&mut *net.rng().unwrap()) {
            Some(&new_root) => {
                // Establish its cluster
                let (cls, _) = explore(net, new_root).unwrap();
                // Mark everything in the cluster as explored
                for c in &cls {
                    unexpl.remove(c);
                }
                clusters.push(cls);
            }
            // The are no unexplored nodes
            None => {
                // Order cluster largest -> smallest
                clusters.sort_by_key(|set| -(set.len() as i64));
                return clusters;
            }
        }
    }
}

/// Perform a breadth-first search over the network, starting with root and looking for target.
/// Returns distance (ie. number of edges) from root to target in the resulting BFS tree or `None`
/// if no path is not found. Returns `Err` if the root or target nodes do not exist.
///
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// use cnetworks::*;
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// match bfs(&net, 1, 3).unwrap() {
///     Some(d) => println!("Edges between 1 and 3: {}", d),
///     None => println!("No path from 1 to 3 found!"),
/// }
/// ```
pub fn bfs(net: &Network, root: usize, target: usize) -> Result<Option<usize>, CNErr> {
    bfs_callback(net, root, target, |_, _, _| ())
}

/// Perform a breadth-first search over the network, starting with root and looking for target.
/// Returns distance (ie. number of edges) from root to target in the resulting BFS tree or `None`
/// if no path is not found. Returns `Err` if the root or target nodes do not exist.
///
/// The `step_callback` is called for every visited node, providing acesss to current node, it's
/// parent (`None` if it's the root) and it's children (linked nodes which have not been visited yet).
///
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// use cnetworks::*;
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// match bfs_callback(
///     &net,
///     1,
///     3,
///     |current: usize, parent: Option<usize>, children: &Vec<&usize>| { /* Do something... */ },
/// ).unwrap() {
///     Some(n) => println!("Found 3 after discovering {:?}", n),
///     None => println!("No path to 3 found!"),
/// }
/// ```
pub fn bfs_callback<F: FnMut(usize, Option<usize>, &Vec<&usize>)>(
    net: &Network,
    root: usize,
    target: usize,
    mut step_callback: F,
) -> Result<Option<usize>, CNErr> {
    // Check if the target and root are valid nodes
    net.get(target)?;
    net.get(root)?;
    let mut discovered: HashMap<usize, Option<usize>> = HashMap::default();
    // The queue stores (node, node parent) pairs
    let mut to_visit: Queue<(usize, Option<usize>)> = Queue::new();
    // Only the root has a `None` parent
    discovered.insert(root, None);
    to_visit.add((root, None)).expect("Queue error");

    while to_visit.size() != 0 {
        let (current, parent) = to_visit.remove().expect("Queue error");
        let children: Vec<_> = net[current]
            .links()
            .keys()
            .filter(|target| !discovered.contains_key(target))
            .collect();
        step_callback(current, parent, &children);
        // Exit condition
        if current == target {
            // Climb back up the tree until the root is reached
            let mut distance: usize = 0;
            let mut parent = parent;
            loop {
                match parent {
                    Some(p) => {
                        parent = discovered[&p];
                        distance += 1;
                    }
                    None => return Ok(Some(distance)),
                }
            }
        }
        for &i in children {
            discovered.insert(i, Some(current));
            to_visit.add((i, Some(current))).expect("Queue error");
        }
    }
    // No path from root to target was found
    Ok(None)
}

/// Explore the network with BFS starting with `root`. Return a tuple of two HashSets: one with the
/// explored nodes (including the root) and another with the unexplored. Returns `Err` if the root
/// does not exist.
///
/// # Panics
/// This function pancis when the queue operations fail.
///
/// # Examples
/// ```
/// use cnetworks::*;
/// let net = Network::new(100, Model::ER { p: 0.05, whole: false }, Weight::None);
/// let (exp, unexp) = explore(&net, 1).unwrap();
/// println!("Available from 1: {:?}", exp);
/// println!("Unavailable from 1: {:?}", unexp);
/// ```
pub fn explore(net: &Network, root: usize) -> Result<(HashSet<usize>, HashSet<usize>), CNErr> {
    net.get(root)?;
    let mut discovered: HashSet<usize> = HashSet::default();
    let mut to_visit: Queue<usize> = Queue::new();
    discovered.insert(root);
    to_visit.add(root).expect("Queue error");
    // Explore the network just like BFS but without specified target
    while to_visit.size() != 0 {
        let current = to_visit.remove().expect("Queue error");
        for &i in net[current].links().keys() {
            if !discovered.contains(&i) {
                discovered.insert(i);
                to_visit.add(i).expect("Queue error");
            }
        }
    }
    // Gather the undiscovered nodes
    let undiscovered: HashSet<usize> = net
        .indexes()
        .filter(|node| !discovered.contains(node))
        .collect();
    Ok((discovered, undiscovered))
}

/// Stitch the network together if it is not whole.
///
/// Connects random elements of smaller clusters to the largest cluster, extending it until it
/// covers the entire network, making the network "whole".
pub fn stitch_together(net: &mut Network) {
    // Get the clusters
    let clusters = clusters(net);
    let mut largest = clusters.first().unwrap().clone();
    for cluster in &clusters[1..] {
        // Connect random node from current cluster to random node from the largest cluster
        let &current = cluster.iter().choose(&mut *net.rng().unwrap()).unwrap();
        let &other = largest.iter().choose(&mut *net.rng().unwrap()).unwrap();
        net.link(current, other).unwrap();
        // Extend the largest cluster with the just-connected nodes
        largest.extend(cluster.iter());
    }
}

/// Simulate spread of information according to the **susceptible-infected** diffusion model.
///
/// On each time step the nodes for which the `infected` flag is `true` will try to infect
/// their closest neighbors with probability given by link weight, setting their
/// "INFECTED" flag.
///
/// Returns the root of the infection.
pub fn spread_si(net: &mut Network, steps: usize) -> usize {
    // Choose random node as origin
    let root = net.indexes().choose(&mut *net.rng().unwrap()).unwrap();
    net.flag(root, "INFECTED", FlagContent::None).unwrap();
    for _ in 0..steps {
        let mut to_infect = Vec::new();
        // Find nodes to infect
        for (index, _) in net.flagged("INFECTED") {
            // Try to infect the neighbors
            for (target, weight) in net[index].links() {
                if net.rng().unwrap().gen::<f64>() < *weight {
                    // Target is getting infected
                    to_infect.push(*target);
                }
            }
        }
        // Infect nodes marked for infection
        for target in to_infect {
            net.flag(target, "INFECTED", FlagContent::None).unwrap();
        }
    }
    root
}
