//! Test linking/unlinking and connecting/disconnecting nodes.
use cnetworks::*;
use std::collections::HashMap;

#[test]
fn basic_link() {
    let mut net = Network::new(10, Model::None, Weight::Constant { c: 1.0 });

    for i in 0..10 {
        for j in 0..10 {
            if let Some(..) = net.get_link(i, i).unwrap() {
                panic!("There should be no link {} to {}", i, j);
            }
        }
    }

    net.link(1, 2).unwrap();
    assert!(net.get_link(1, 2).unwrap().unwrap() == 1.0);
    assert!(net.get_link(2, 1).unwrap().unwrap() == 1.0);

    net.link_exact(2, 3, 0.25).unwrap();
    assert!(net.get_link(2, 3).unwrap().unwrap() == 0.25);
    assert!(net.get_link(3, 2).unwrap().unwrap() == 0.25);

    assert_eq!(
        net.link(1, 100).unwrap_err(),
        Error::NoSuchNode { idx: 100 }
    );
    assert_eq!(
        net.get_link(1, 100).unwrap_err(),
        Error::NoSuchNode { idx: 100 }
    );
}

#[test]
fn basic_unlink() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });

    net.link(0, 1).unwrap();
    assert_eq!(net.unlink_safe(0, 1).unwrap(), None);
    assert_eq!(net.unlink(0, 1).unwrap().unwrap(), 1.0);
    assert_eq!(net.get_link(0, 1).unwrap(), None);
    assert_eq!(net.get_link(1, 0).unwrap(), None);

    assert_eq!(
        net.unlink(0, 100).unwrap_err(),
        Error::NoSuchNode { idx: 100 }
    );
    assert_eq!(
        net.unlink_safe(0, 100).unwrap_err(),
        Error::NoSuchNode { idx: 100 }
    );

    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    net.link(0, 2).unwrap();
    assert_eq!(net.unlink_safe(0, 1).unwrap().unwrap(), 1.0);
    assert_eq!(net.get_link(0, 1).unwrap(), None);
    assert_eq!(net.get_link(1, 0).unwrap(), None);
}

#[test]
fn basic_disconnect() {
    let mut net = Network::new(
        3,
        Model::ER {
            p: 1.0,
            whole: true,
        },
        Weight::Constant { c: 1.0 },
    );

    for (target, weight) in net.disconnect_safe(0).unwrap().unwrap() {
        assert!(vec![1, 2].contains(&target));
        assert_eq!(weight, 1.0);
    }
    assert_eq!(net.get_link(0, 1).unwrap(), None);
    assert_eq!(net.disconnect(0).unwrap(), HashMap::new());
    assert_eq!(net.disconnect(1).unwrap()[&2], 1.0);
    assert_eq!(net.disconnect(2).unwrap(), HashMap::new());
    assert_eq!(
        net.disconnect(100).unwrap_err(),
        Error::NoSuchNode { idx: 100 }
    );
}
