//! Test linking/unlinking and connecting/disconnecting nodes.
use cnetworks::*;
use std::collections::HashSet;
use std::iter::FromIterator;

#[test]
fn test_bfs() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(
        bfs(&net, 0, 2).unwrap().unwrap(),
        HashSet::from_iter(vec![0, 1, 2].iter().cloned())
    );
    net.unlink(0, 1).unwrap();
    assert_eq!(bfs(&net, 0, 2).unwrap(), None);
}

#[test]
fn test_explore() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(
        explore(&net, 0).unwrap(),
        (
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::new()
        )
    );
    assert_eq!(
        explore(&net, 1).unwrap(),
        (
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::new()
        )
    );
    assert_eq!(
        explore(&net, 2).unwrap(),
        (
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::new()
        )
    );
    net.unlink(1, 2).unwrap();
    assert_eq!(
        explore(&net, 0).unwrap(),
        (
            HashSet::from_iter(vec![0, 1].iter().cloned()),
            HashSet::from_iter(vec![2].iter().cloned())
        )
    );
    assert_eq!(
        explore(&net, 1).unwrap(),
        (
            HashSet::from_iter(vec![0, 1].iter().cloned()),
            HashSet::from_iter(vec![2].iter().cloned())
        )
    );
    assert_eq!(
        explore(&net, 2).unwrap(),
        (
            HashSet::from_iter(vec![2].iter().cloned()),
            HashSet::from_iter(vec![1, 0].iter().cloned())
        )
    );
}

#[test]
fn test_clusters() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(
        net.clusters(),
        vec![
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::from_iter(vec![3].iter().cloned())
        ]
    );
    net.link(1, 3).unwrap();
    assert_eq!(
        net.clusters(),
        vec![HashSet::from_iter(vec![0, 1, 2, 3].iter().cloned()),]
    );
}

#[test]
fn test_spread_si() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    net.link(2, 3).unwrap();
    let root = spread_si(&mut net, 1);
    if root == 0 {
        assert_eq!(
            net.get_infected(),
            HashSet::from_iter(vec![0, 1].iter().cloned())
        )
    } else if root == 1 {
        assert_eq!(
            net.get_infected(),
            HashSet::from_iter(vec![0, 1, 2].iter().cloned())
        )
    } else if root == 2 {
        assert_eq!(
            net.get_infected(),
            HashSet::from_iter(vec![1, 2, 3].iter().cloned())
        )
    } else if root == 3 {
        assert_eq!(
            net.get_infected(),
            HashSet::from_iter(vec![2, 3].iter().cloned())
        )
    } else {
        panic!("Root should be in the network, {} is not", root);
    }
    net.clear();
    let root = spread_si(&mut net, 3);
    assert!(vec![0, 1, 2, 3].contains(&root));
    assert_eq!(
        net.get_infected(),
        HashSet::from_iter(vec![0, 1, 2, 3].iter().cloned())
    )
}

#[test]
fn test_stitch() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    stitch_together(&mut net);
    assert_eq!(
        explore(&net, 0).unwrap(),
        (
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::new()
        )
    );
    assert_eq!(
        explore(&net, 1).unwrap(),
        (
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::new()
        )
    );
    assert_eq!(
        explore(&net, 2).unwrap(),
        (
            HashSet::from_iter(vec![0, 1, 2].iter().cloned()),
            HashSet::new()
        )
    );
}
