//! This is an example of using the library to find the experimental critical order parameter (ie.
//! the average degree distribution) when the phenomenon of **percolation** occurs.
//!
//! Percolation in a random graph (or a complex network using the Erods-Renyi model) occurs when
//! the largest cluster (set of connected nodes) becomes **macroscopic** - its size is comparable
//! to the size of the network.
//!
//! In this example we will calculate the average degree, largest cluster size and average size of
//! clusters without the largest one of ER networks for multiple probabilities and save them to a
//! text file, making it possible to graph with your tool of choice.
//!
//! See the associated Jupyter Notebook for example of data analisys.

use cnetworks::*;
use std::fs::File;
use std::io::prelude::*;
use std::path::Path;
use std::thread;

fn calculate(deg_expected: f64, file: &mut File, nsize: usize, nsamples: usize) {
    println!("=> [Size {}] Starting {}", nsize, deg_expected);
    let p = deg_expected / ((nsize - 1) as f64);
    let mut net = Network::new(nsize, Model::None, Weight::Constant { c: 1.0 });
    for _ in 0..nsamples {
        net.disconnect_all();
        Network::init_er(&mut net, p, false);
        let mut avg_othersize = 0.0;
        let clsters = net.clusters();
        for clst in &clsters[1..] {
            avg_othersize += clst.len() as f64;
        }
        avg_othersize /= (clsters.len() - 1) as f64;
        let to_write = format!(
            "{} {} {} {}\n",
            deg_expected,
            net.avg_deg(),
            clsters[0].len(),
            avg_othersize
        );
        file.write_all(to_write.as_bytes()).unwrap();
    }
    println!("=> [Size {}] Done {}", nsize, deg_expected);
}

fn enqueue_thread(
    threads: &mut Vec<thread::JoinHandle<()>>,
    start: f64,
    stop: f64,
    step: f64,
    nsize: usize,
    nsamples: usize,
) {
    threads.push(thread::spawn(move || {
        let name = format!(
            "target/size{}samples{}-{}-{}.txt",
            nsize,
            nsamples,
            start + step,
            stop
        );
        let path = Path::new(&name);
        let mut file = File::create(&path).unwrap();
        let i_max = ((stop - start) / step).round() as i64;
        for i in 1..=i_max {
            let deg = start + i as f64 * step;
            calculate(deg, &mut file, nsize, nsamples);
        }
        println!("=> [Size {}] Thread {}-{} done", nsize, start + step, stop,);
    }));
}

fn main() {
    let mut threads: Vec<thread::JoinHandle<()>> = vec![];

    // Size 1000 - full deal form 0 to 5
    enqueue_thread(&mut threads, 0.0, 0.5, 0.1, 1000, 100);
    enqueue_thread(&mut threads, 0.5, 0.75, 0.05, 1000, 100);
    enqueue_thread(&mut threads, 0.75, 1.0, 0.025, 1000, 100);
    enqueue_thread(&mut threads, 1.0, 1.25, 0.025, 1000, 100);
    enqueue_thread(&mut threads, 1.25, 1.5, 0.05, 1000, 100);
    enqueue_thread(&mut threads, 1.5, 2.0, 0.1, 1000, 100);
    enqueue_thread(&mut threads, 2.0, 3.5, 0.25, 1000, 100);
    enqueue_thread(&mut threads, 3.5, 5.0, 0.25, 1000, 100);

    // Smallers sizes - just the wedge around 0.9-1.8, but lots of samples
    for size in vec![100, 200, 500] {
        enqueue_thread(&mut threads, 0.9, 1.1, 0.025, size, 10000);
        enqueue_thread(&mut threads, 1.1, 1.3, 0.025, size, 10000);
        enqueue_thread(&mut threads, 1.3, 1.5, 0.025, size, 10000);
        enqueue_thread(&mut threads, 1.5, 1.65, 0.025, size, 10000);
        enqueue_thread(&mut threads, 1.65, 1.8, 0.025, size, 10000);
    }

    // Larger sizes - zoom on the wedge again
    for size in vec![1000, 2000, 5000, 10000] {
        enqueue_thread(&mut threads, 0.9, 1.0, 0.025, size, 1000);
        enqueue_thread(&mut threads, 1.0, 1.1, 0.025, size, 1000);
        enqueue_thread(&mut threads, 1.1, 1.2, 0.025, size, 1000);
        enqueue_thread(&mut threads, 1.2, 1.3, 0.025, size, 1000);
        enqueue_thread(&mut threads, 1.3, 1.4, 0.025, size, 1000);
    }

    for th in threads {
        th.join().unwrap();
    }
}
