//! This is a library of tools for creating and manipulating complex networks.
//!
//! From [Wikipedia](https://en.wikipedia.org/wiki/Complex_network):
//!
//! > Complex network is a graph-like structure with non-trivial characteristics.
//! > A complex network is a graph (network) with non-trivial topological features—features that do not
//! > occur in simple networks such as lattices or random graphs but often occur in networks representing
//! > real systems. The study of complex networks is a young and active area of scientific
//! > research inspired largely by empirical findings of real-world networks such
//! > as computer networks, biological networks, technological networks, brain networks, climate networks
//! > and social networks.

extern crate rand;
mod algorithms;
mod enums;
mod link;
mod node;

use node::Node;
use rand::prelude::*;
use std::collections::{HashMap, HashSet};

pub use self::algorithms::*;
pub use self::enums::*;
pub use self::link::*;

/// `Network` is a graph-like data structure, containing a `Vec` of `Node` objects.
#[derive(Debug, Clone)]
pub struct Network {
    /// The main data structure.
    nodes: Vec<Node>,
    /// The model used to initialize links between nodes.
    model: NetworkModel,
    /// The distribution followed by the link weights.
    weight: LinkWeight,
}

impl Network {
    /// Create a new network of given size and desired model. Use `NetworkModel::None` if you want
    /// a network with no connections.
    /// # Examples
    /// Creating a network of chosen model and link weight:
    /// ```
    /// let net = Network::new(20, NetworkModel::ER { p: 0.4, true }, LinkWeight::Uniform);
    /// println!("{:?}", net);
    /// ```
    /// Creating a network with no links and establishing them "by hand":
    /// ```
    /// let net = Network::new(10, NetworkModel::None, LinkWeight::Constant { c: 0.25 });
    /// net.link(1, 2);
    /// net.link(4, 7);
    /// println!("{:?}", net);
    /// ```
    pub fn new(size: usize, model: NetworkModel, weight: LinkWeight) -> Self {
        // Check for value correctness
        if size == 0 {
            panic!("Attempted creation of network with no nodes.");
        }
        if let LinkWeight::Constant { c } = weight {
            if c <= 0. || c > 1. {
                panic!("Constant link weight cannot be {}", c);
            }
        }
        // Create the network
        let mut net = Network {
            model: NetworkModel::None,
            nodes: Vec::new(),
            weight,
        };
        // Fill in the nodes
        for i in 0..size {
            net.nodes.push(Node {
                index: i,
                links: HashMap::new(),
                infected: false,
            });
        }
        // Initialize links according to the desired model
        match model {
            NetworkModel::ER { p, whole } => Network::init_er(&mut net, p, whole),
            NetworkModel::BA { m0, m } => Network::init_ba(&mut net, m0, m),
            NetworkModel::None => (),
        }
        net
    }

    /// Initialize (or reinitialize) the network's links according to the Erdos-Renyi model.
    /// See the [NetworkModel](enum.NetworkModel.html) for more detailed model explanation.
    ///
    /// If `whole` is `true` then the network is artificially stitched together after
    /// initialization, otherwise there is no guarantee that there are no 'outsiders' or even
    /// separate networks.
    ///
    /// Beware that the network is **not** cleared before linking.
    pub fn init_er(net: &mut Network, p: f64, whole: bool) {
        let net_len = net.len();
        if p <= 0. || p > 1. {
            panic!("The probability of connecting cannot be {}", p);
        }
        let mut rng = rand::thread_rng();
        for i in 0..net_len {
            for j in i + 1..net_len {
                if rng.gen::<f64>() <= p {
                    net._link(i, j, &mut rng)
                }
            }
        }
        if whole {
            algorithms::stitch_together(net);
        }
        net.model = NetworkModel::ER { p, whole };
    }

    /// Initialize (or reinitialize) the network's links according to the Barabasi-Albert model.
    /// See the [NetworkModel](enum.NetworkModel.html) for more detailed model explanation.
    ///
    /// The links *are* cleared before re-linking because of the nautre of initialization algorithm
    pub fn init_ba(net: &mut Network, m0: usize, m: usize) {
        let net_len = net.len();
        if m0 == 0 || m == 0 || m > m0 || m0 > net_len {
            panic!("Incorrrect model parameters: m0 = {}, m = {}", m0, m);
        }
        net.disconnect_all();
        let mut rng = rand::thread_rng();
        // Initial cluster - connect everything to everything
        for i in 0..m0 {
            for j in i + 1..m0 {
                net._link(i, j, &mut rng);
            }
        }
        // Track the total degree for speed
        let mut total_deg = m0 * m0;
        // The rest is attached one by one
        for i in m0..net_len {
            let mut connections_made = 0;
            // Make exactly m connections when attaching
            while connections_made != m {
                // Try connecting to the already established nodes
                for j in 0..i {
                    // Do not connect to the same node twice
                    if !net.nodes[i].links.contains_key(&j) {
                        // Probability of connecting to any node is its deg / total deg
                        let p = net.nodes[j].deg() as f64 / total_deg as f64;
                        if rng.gen::<f64>() <= p {
                            net._link(i, j, &mut rng);
                            connections_made += 1;
                            // Total degree increases by one (current node does not count)
                            total_deg += 1;
                        }
                    }
                }
            }
            // The newly attached node adds m of their own degrees
            total_deg += m;
        }
        net.model = NetworkModel::BA { m0, m };
    }

    /// Return the network size, ie. the total number of nodes.
    fn len(&self) -> usize {
        self.nodes.len()
    }

    /// Return the total number of edges in the network.
    pub fn edges(&self) -> usize {
        let mut edges = 0;
        for node in &self.nodes {
            edges += node.deg();
        }
        edges / 2
    }

    /// Calculates the arithmetical average of vertex degrees (ie. number of closest neighbors)
    /// over the network.
    pub fn avg_deg(&self) -> f64 {
        let mut sum = 0;
        for node in &self.nodes {
            sum += node.deg();
        }
        sum as f64 / self.len() as f64
    }

    /// Returns the degree distribution of the network as a `HashMap` of histogram bins where key -
    /// degree, value - number of occurences in the network.
    pub fn deg_distr(&self) -> HashMap<usize, usize> {
        let mut bins: HashMap<usize, usize> = HashMap::new();
        for node in &self.nodes {
            let deg = node.deg();
            match bins.get_key_value(&deg) {
                // If there already is such bin then increment it by 1
                Some((&key, &value)) => bins.insert(key, value + 1),
                // If there isn't create one with single occurence
                None => bins.insert(deg, 1),
            };
        }
        bins
    }

    /// Sets the `infected` on all nodes to `false`.
    pub fn clear(&mut self) {
        for node in &mut self.nodes {
            node.infected = false;
        }
    }

    /// Returns a `HashSet` with indexes of healthy (not infected) nodes.
    pub fn get_healthy(&self) -> HashSet<usize> {
        let mut hlth: HashSet<usize> = HashSet::new();
        for node in &self.nodes {
            if !node.infected {
                hlth.insert(node.index);
            }
        }
        hlth
    }

    /// Returns a `HashSet` with indexes of infected nodes.
    pub fn get_infected(&self) -> HashSet<usize> {
        let mut inf: HashSet<usize> = HashSet::new();
        for node in &self.nodes {
            if node.infected {
                inf.insert(node.index);
            }
        }
        inf
    }
}
