//! This module contains various enums used for network classification.

/// `NetworkModel` decides the algorithm used to connect the nodes during `Network`'s
/// initialization.
#[derive(Debug, Clone)]
pub enum NetworkModel {
    /// The Erdos–Renyi random network model.
    ER {
        /// The probability of connecting any given pair of nodes.
        p: f64,
        /// Should the network be in one piece
        whole: bool,
    },
    /// The Barabasi–Albert preferential attachment model.
    BA {
        /// The initial number of clustered nodes.
        m0: usize,
        /// The number of nodes added in each step during network creation.
        /// Keep in mind that this should be strictly less or equal m0 and the Barabasi-Albert
        /// initialization fuction **will panic** if this is not the case.
        m: usize,
    },
    /// Placeholder for a network with no connections.
    None,
}

/// `LinkWeight` determines the weight (ie. the probability of infection) of each link in the
/// network.
#[derive(Debug, Clone)]
pub enum LinkWeight {
    /// The probability should be constant end equalt to `c`. Keep in mind that 0 < c <= 1, the
    /// `Network` constructor **will panic** if that is not the case.
    Constant { c: f64 },
    /// A weight sampled uniformly from (0, 1).
    Uniform,
}
