#[cfg(feature = "arrayvec")]
mod arrayvec;
mod float;
mod std_impls;
#[cfg(feature = "uuid")]
mod uuid;

use float::WellBehavedF64;
use std::fmt;
use std::iter;
use std::rc::Rc;
use thiserror::Error;

#[derive(Debug, Error)]
pub enum Error {
    #[error("Expected value")]
    ExpectedValue,
    #[error("Expected node")]
    ExpectedNode,
    // TODO include names of actual and expected types
    #[error("Type error")]
    TypeError,
    #[error("Expected integer")]
    ExpectedInt,
    #[error("Variable not found")]
    VarNotFound,
    #[error("Function not found")]
    FnNotFound,
    #[error("Cannot execute a path on a value")]
    PathOnValue,
    #[error("Input is empty in {0}")]
    Empty(&'static str),
    #[error("Conversion failed: {0}")]
    Conversion(#[from] Box<dyn std::error::Error>),
}

pub type Result<T> = std::result::Result<T, Error>;

pub type Node<'a> = &'a (dyn Queryable + 'a);

#[derive(Debug, Clone, PartialEq, PartialOrd, Eq, Ord, Hash)]
pub enum Value {
    String(String),
    Int(i64),
    Float(WellBehavedF64),
    Bool(bool),
    Array(Array),
}

pub struct ValueIter<'a>(pub Box<dyn Iterator<Item = Value> + 'a>);

impl<'a> ValueIter<'a> {
    pub fn empty() -> Self {
        ValueIter(Box::new(iter::empty()))
    }

    pub fn once<V: Into<Value>>(val: V) -> Self {
        ValueIter(Box::new(iter::once(val.into())))
    }

    pub fn from_values<T: 'a>(values: impl IntoIterator<Item = T> + 'a) -> Self
    where
        T: Into<Value>,
    {
        ValueIter(Box::from(values.into_iter().map(|v| v.into())))
    }
}

impl<'a> Iterator for ValueIter<'a> {
    type Item = Value;
    fn next(&mut self) -> Option<Self::Item> {
        self.0.next()
    }
}

pub struct NodeIter<'a>(pub Box<dyn Iterator<Item = Node<'a>> + 'a>);

impl<'a> NodeIter<'a> {
    pub fn empty() -> NodeIter<'a> {
        NodeIter(Box::new(iter::empty()))
    }

    pub fn from_queryables<Q>(queryables: impl IntoIterator<Item = &'a Q> + 'a) -> Self
    where
        Q: Queryable + 'a,
    {
        NodeIter(Box::from(queryables.into_iter().map(|q| q as _)))
    }
}

impl<'a> Iterator for NodeIter<'a> {
    type Item = Node<'a>;
    fn next(&mut self) -> Option<Self::Item> {
        self.0.next()
    }
}

#[derive(Debug)]
pub enum NodeOrValue<'a> {
    Node(Node<'a>),
    Value(Value),
}

impl<'a> NodeOrValue<'a> {
    pub fn try_into_int(self) -> Result<i64> {
        self.try_into_value()
            .map_err(|_| Error::ExpectedInt)
            .and_then(Value::try_into_int)
    }

    pub fn try_into_value(self) -> Result<Value> {
        match self {
            NodeOrValue::Node(node) => {
                if let Some(data) = node.data() {
                    Ok(data)
                } else {
                    Err(Error::ExpectedValue)
                }
            }
            NodeOrValue::Value(v) => Ok(v),
        }
    }

    pub fn try_into_node(self) -> Result<Node<'a>> {
        match self {
            NodeOrValue::Node(n) => Ok(n),
            NodeOrValue::Value(_) => Err(Error::ExpectedNode),
        }
    }
}

impl<'a> From<Value> for NodeOrValue<'a> {
    fn from(v: Value) -> Self {
        NodeOrValue::Value(v)
    }
}

impl<'a> From<Node<'a>> for NodeOrValue<'a> {
    fn from(n: Node<'a>) -> Self {
        NodeOrValue::Node(n)
    }
}

impl<'a> fmt::Display for NodeOrValue<'a> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            NodeOrValue::Node(node) => write!(f, "{}", node),
            NodeOrValue::Value(value) => write!(f, "{}", value),
        }
    }
}

pub struct NodeOrValueIter<'a>(Box<dyn Iterator<Item = Result<NodeOrValue<'a>>> + 'a>);

impl<'a> NodeOrValueIter<'a> {
    pub fn empty() -> Self {
        NodeOrValueIter(Box::new(iter::empty()))
    }

    pub fn one(item: Result<NodeOrValue<'a>>) -> Self {
        NodeOrValueIter(Box::new(iter::once(item)))
    }

    pub fn one_node(node: Node<'a>) -> Self {
        Self::from_raw(iter::once(node).map(|n| Ok(n.into())))
    }

    pub fn one_value(value: Value) -> Self {
        Self::from_raw(iter::once(value).map(|v| Ok(v.into())))
    }

    pub fn from_queryables<I, Q>(nodes: I) -> Self
    where
        I: IntoIterator<Item = &'a Q> + 'a,
        Q: Queryable + 'a,
    {
        NodeOrValueIter(Box::new(
            nodes.into_iter().map(|node| Ok(NodeOrValue::Node(node))),
        ))
    }

    pub fn from_nodes<I>(nodes: I) -> Self
    where
        I: IntoIterator<Item = &'a dyn Queryable> + 'a,
    {
        NodeOrValueIter(Box::new(
            nodes.into_iter().map(|node| Ok(NodeOrValue::Node(node))),
        ))
    }

    pub fn from_values<I>(values: I) -> Self
    where
        I: IntoIterator + 'a,
        I::Item: Into<Value>,
    {
        NodeOrValueIter(Box::new(
            values
                .into_iter()
                .map(|value| Ok(NodeOrValue::Value(value.into()))),
        ))
    }

    pub fn from_raw<I>(iter: I) -> Self
    where
        I: IntoIterator<Item = Result<NodeOrValue<'a>>> + 'a,
    {
        NodeOrValueIter(Box::new(iter.into_iter()))
    }
}

impl<'a> Iterator for NodeOrValueIter<'a> {
    type Item = Result<NodeOrValue<'a>>;
    fn next(&mut self) -> Option<Self::Item> {
        self.0.next()
    }
}

pub trait Queryable {
    fn keys(&self) -> ValueIter<'_> {
        ValueIter::empty()
    }
    fn member<'f>(&self, _: &'f Value) -> Option<Node<'_>> {
        None
    }
    fn all(&self) -> NodeOrValueIter<'_> {
        NodeOrValueIter::empty()
    }
    fn descendants(&self) -> NodeOrValueIter<'_> {
        if self.all().next().is_some() {
            NodeOrValueIter::from_raw(self.all().chain(self.all().flat_map(|node| {
                if let Ok(node) = node {
                    match node {
                        NodeOrValue::Node(node) => node.descendants(),
                        NodeOrValue::Value(value) => NodeOrValueIter::one_value(value),
                    }
                } else {
                    NodeOrValueIter::empty()
                }
            })))
        } else {
            self.all()
        }
    }
    fn name(&self) -> &'static str;
    fn data(&self) -> Option<Value> {
        None
    }
}

#[derive(Debug, Clone, PartialEq, PartialOrd, Eq, Ord, Hash)]
pub struct Array(pub Rc<[Value]>);

impl Array {
    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }
}

impl IntoIterator for Array {
    type Item = Value;
    type IntoIter = ArrayIter;
    fn into_iter(self) -> ArrayIter {
        ArrayIter {
            current: 0,
            array: self,
        }
    }
}

pub struct ArrayIter {
    current: usize,
    array: Array,
}

impl Iterator for ArrayIter {
    type Item = Value;
    fn next(&mut self) -> Option<Value> {
        if self.current == self.array.len() {
            None
        } else {
            let index = self.current;
            self.current += 1;
            Some(self.array.0[index].clone())
        }
    }
}

impl fmt::Display for Array {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.write_str("{")?;
        let mut it = self.clone().into_iter().peekable();
        while let Some(val) = it.next() {
            let mut format = |args| {
                if it.peek().is_some() {
                    write!(f, "{}, ", args)
                } else {
                    write!(f, "{}", args)
                }
            };
            if let Value::String(s) = val {
                format(format_args!(r#""{}""#, s))
            } else {
                format(format_args!("{}", val))
            }?;
        }
        f.write_str("}")
    }
}

impl Value {
    pub fn try_into_int(self) -> Result<i64> {
        if let Value::Int(i) = self {
            Ok(i)
        } else {
            Err(Error::ExpectedInt)
        }
    }
}

impl fmt::Display for Value {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Value::String(v) => write!(f, "{}", v),
            Value::Int(v) => write!(f, "{}", v),
            Value::Bool(v) => write!(f, "{}", v),
            Value::Array(v) => write!(f, "{}", v),
            Value::Float(v) => write!(f, "{}", v),
        }
    }
}

impl<'a> fmt::Display for Node<'a> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        if let Some(data) = self.data() {
            write!(f, "{}", data)
        } else {
            write!(f, "[{}]", self.name())
        }
    }
}

impl<'a> fmt::Debug for &'a dyn Queryable {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "[{} {:?}]", self.name(), self.data())
    }
}
