//! Scans a progressive JPEG file to find which byte ranges of the file are critical for displaying it at key stages.
//! This knowledge can be used to serve JPEGs optimally over HTTP/2 connections. This library can generate cf-priority-change headers
//! compatible with [prioritization syntax used by Cloudflare](https://blog.cloudflare.com/parallel-streaming-of-progressive-images/).

use std::fmt::Write;

mod error;
pub use crate::error::*;

/// For advanced usage, low-level access to the basic JPEG structure
pub mod jpeg;

#[cfg(feature = "gif")]
mod gif;
mod png;

#[cfg(target_arch = "wasm32")]
use wasm_bindgen::prelude::*;

/// Key positions in a progressive image file
///
/// ```rust,no_run
/// # let input_file = vec![];
/// soos::Scans::from_file(&input_file)?.cf_priority_change_headers()?;
/// # Ok::<_, soos::Error>(())
/// ```
#[derive(Debug, Clone, Default, PartialEq)]
pub struct Scans {
    /// Byte position where metadata ends.
    /// This many bytes are preceeding start of pixel data.
    /// It's usually <200 bytes, unless the image has color profiles or other bloat.
    pub metadata_end: Option<usize>,
    /// All metadata + minimum amount of data to make browsers render _anything_
    /// (in case they don't reserve space based on metadata)
    pub frame_render_start: Option<usize>,
    /// Byte position where the first (lowest-quality) progressive scan ends.
    /// This many bytes are needed to display anything on screen.
    /// It's usually 12-15% of the file size.
    pub first_scan_end: Option<usize>,
    /// Byte position where most of ok-quality pixel data ends.
    /// This many bytes are needed to display a good-enough image.
    /// It's usually about 50% of the file size.
    pub good_scan_end: Option<usize>,
    /// Size of the whole image file, in bytes
    pub file_size: usize,
}

#[cfg(target_arch = "wasm32")]
#[global_allocator]
static ALLOC: wee_alloc::WeeAlloc = wee_alloc::WeeAlloc::INIT;

#[cfg(target_arch = "wasm32")]
#[wasm_bindgen]
pub fn cf_priority_change_header_wasm(image: &[u8]) -> Option<String> {
    cf_priority_change_header(image).ok()
}

impl Scans {
    /// Analyze an image file to find byte ranges of its metadata and progressive scans
    pub fn from_file(input_file: &[u8]) -> Result<Self> {
        match input_file {
            [0xff, ..] => crate::jpeg::scans(input_file),
            #[cfg(feature = "gif")]
            [b'G', ..] => crate::gif::scans(input_file),
            [0x89, ..] => crate::png::scans(input_file),
            _ => Err(Error::Unsupported),
        }
    }

    /// Assumes the priorities are:
    ///
    ///  * 50 = critical js/css, fonts + image metadata
    ///  * 30 = regular js/css, followed by other images + JPEG DC
    ///  * 20 = low-priority image bodies
    ///  * 10 = idle
    ///
    /// Returns `cf-priority` and `cf-priority-change` header values, respectively.
    pub fn cf_priority_change_headers(&self) -> Result<(String, String)> {
        let mut metadata_end = self.metadata_end.unwrap_or(0);
        let is_progressive = self.first_scan_end.map_or(false, |len| len < self.file_size / 2);

        // if there's a fat color profile or Adobe Garbage,
        // then sending "just" the metadata is not fast, and shouldn't be prioritized.
        let fat_metadata_limit = ((self.file_size / 8) + 180).min(2000);
        let fat_frame_start_limit = ((self.file_size / 8) + 500).min(8000);

        // if the metadata can cheaply include first frame/scan info, then send both at once (reduce H/2 framing overhead)
        let rendered_anything = self.frame_render_start.or(self.first_scan_end).or(self.good_scan_end);
        if let Some(rendered_anything) = rendered_anything {
            if rendered_anything < fat_metadata_limit && rendered_anything < metadata_end + metadata_end / 8 + 100 {
                metadata_end = rendered_anything;
            }
        }

        let mut chunks = Vec::with_capacity(8);

        // This is important, because it decides when the whole image starts sending
        let image_priority = if is_progressive && metadata_end < fat_metadata_limit {
            "50/0" // fast and worth accelerating
        } else if self.file_size < 1200 || is_progressive || rendered_anything.map_or(false, |n| n < 2000) {
            "30/1" // at least we can show something quick
        } else {
            "21/n" // lost cause: baseline image with bloated metadata
        };

        chunks.push((metadata_end, image_priority));

        // Browsers don't always reserve space based on metadata availability alone,
        // so here's trying again, with more data
        if let Some(frame_render_start) = self.frame_render_start {
            if frame_render_start > metadata_end && self.first_scan_end.map_or(true, |dc| frame_render_start < dc) {
                let start_render_priority = if frame_render_start < fat_frame_start_limit {
                    if frame_render_start < 1000 { "50/1" } else { "40/1" }
                } else { "30/n" };
                chunks.push((frame_render_start, start_render_priority));
            }
        }

        if let Some(first_scan_end) = self.first_scan_end {
            // small DC can be downloaded in one go, to save on re-rendering
            chunks.push((first_scan_end, if first_scan_end < 25000 {"30/0"} else {"30/1"}));
        }

        if let Some(good_scan_end) = self.good_scan_end {
            chunks.push((good_scan_end, if good_scan_end < 100_000 { "20/1" } else { "20/n" }));
        }

        let rendered_already = self.first_scan_end.is_some() || self.good_scan_end.is_some();
        let bytes_left = self.file_size.saturating_sub(self.good_scan_end.or(self.first_scan_end).unwrap_or(0));
        let is_big = bytes_left > 80_000;
        let is_tiny = bytes_left < 1_000;

        chunks.push((self.file_size, if rendered_already {
            if is_big { "10/n" } else { "10/1" } // if it's on screen, it's not urgent to send anything more
        } else {
            if is_tiny { "30/1" } else if is_big { "20/n" } else { "20/1" }
        }));

        let pri_header = String::from(chunks[0].1);
        let mut change_header = String::with_capacity(128);

        // cf-priority handles chunk at pos 0. There may be more redundant ones.
        let mut min_pos = 0;
        chunks.retain(|&(pos, _)| {
            let res = pos > min_pos;
            min_pos = pos + 20; // there's no point changing priority for 20 bytes, H/2 frame takes half of that
            res
        });

        for c in chunks.windows(2) {
            // chunks describe themselves, but priority-change header describes the next one
            let pri = c[1].1;
            let pos = c[0].0;
            if !change_header.is_empty() {
                change_header.push(',');
            }
            let _ = write!(&mut change_header, "{}:{}", pos, pri);
        }

        if change_header.is_empty() {
            return Err(Error::Format("Can't find useful scans"));
        }
        Ok((String::from(pri_header), change_header))
    }
}

#[cfg(test)]
fn s(a: &str, b: &str) -> (String, String) { (a.into(), b.into()) }

#[test]
fn test_baseline() {
    let res = (Scans {
        metadata_end: None,
        frame_render_start: None,
        first_scan_end: None,
        good_scan_end: None,
        file_size: 100_000}).cf_priority_change_headers();
    assert!(res.is_err(), "{:?}", res);

    // regular baseline image
    assert_eq!(s("30/1", "181:20/n"), (Scans {
        metadata_end: Some(101),
        frame_render_start: Some(181),
        first_scan_end: None,
        good_scan_end: None,
        file_size: 100_000}).cf_priority_change_headers().unwrap());

    // tiny image
    assert_eq!(s("30/1", "101:20/1"), (Scans {
        metadata_end: Some(101),
        frame_render_start: None,
        first_scan_end: None,
        good_scan_end: None,
        file_size: 1000}).cf_priority_change_headers().unwrap());

    // fat metadata
    assert_eq!(s("21/n", "9999:20/n"), (Scans {
        metadata_end: Some(9999),
        frame_render_start: None,
        first_scan_end: None,
        good_scan_end: None,
        file_size: 100_000}).cf_priority_change_headers().unwrap());

    // relatively fat metadata
    assert_eq!(s("21/n", "1000:20/1"), (Scans {
        metadata_end: Some(1000),
        frame_render_start: None,
        first_scan_end: None,
        good_scan_end: None,
        file_size: 3000}).cf_priority_change_headers().unwrap());
}

#[test]
fn test_progressive() {
    assert_eq!(s("50/0", "1000:30/0,10000:20/n,100000:10/n"), (Scans {
        metadata_end: Some(1_000),
        frame_render_start: None,
        first_scan_end: Some(10_000),
        good_scan_end: Some(100_000),
        file_size: 200_000}).cf_priority_change_headers().unwrap());

    // fat metadata
    assert_eq!(s("30/1", "4000:30/0,10000:20/n,100000:10/n"), (Scans {
        metadata_end: Some(4_000),
        frame_render_start: None,
        first_scan_end: Some(10_000),
        good_scan_end: Some(100_000),
        file_size: 200_000}).cf_priority_change_headers().unwrap());

    // fat DC
    assert_eq!(s("50/0", "1000:30/1,50000:20/n,100000:10/n"), (Scans {
        metadata_end: Some(1_000),
        frame_render_start: None,
        first_scan_end: Some(50_000),
        good_scan_end: Some(100_000),
        file_size: 200_000}).cf_priority_change_headers().unwrap());

    // small good scan
    assert_eq!(s("50/0", "1000:30/0,10000:20/1,11000:10/n"), (Scans {
        metadata_end: Some(1_000),
        frame_render_start: None,
        first_scan_end: Some(10_000),
        good_scan_end: Some(11_000),
        file_size: 200_000}).cf_priority_change_headers().unwrap());
}
