use std::cmp;
use std::mem;
use std::iter;
use std::io::BufRead;

use super::common::{Block, Frame};

mod decoder;
pub use self::decoder::{
    PLTE_CHANNELS, StreamingDecoder, Decoded, DecodingError, DecodingFormatError,
    Version
};

pub fn read_info<R: BufRead>(r: R) -> Result<Decoder<R>, DecodingError> {
    Decoder::with_no_init(r, StreamingDecoder::with_options()).init()
}

struct ReadDecoder<R: BufRead> {
    reader: R,
    decoder: StreamingDecoder,
    at_eof: bool
}

impl<R: BufRead> ReadDecoder<R> {
    fn decode_next(&mut self) -> Result<Option<Decoded>, DecodingError> {
        while !self.at_eof {
            let (consumed, result) = {
                let buf = self.reader.fill_buf()?;
                if buf.len() == 0 {
                    return Err(DecodingError::format(
                        "unexpected EOF"
                    ))
                }
                self.decoder.update(buf)?
            };
            self.reader.consume(consumed);
            match result {
                Decoded::Nothing => (),
                Decoded::BlockStart(Block::Trailer) => {
                    self.at_eof = true
                },
                result => return Ok(unsafe{
                    // FIXME: #6393
                    Some(mem::transmute::<Decoded, Decoded>(result))
                }),
            }
        }
        Ok(None)
    }
}

#[allow(dead_code)]
/// GIF decoder
pub struct Decoder<R: BufRead> {
    decoder: ReadDecoder<R>,
    current_frame: Frame,
    buffer: Vec<u8>,
}

impl<R> Decoder<R> where R: BufRead {
    fn with_no_init(reader: R, decoder: StreamingDecoder) -> Decoder<R> {
        Decoder {
            decoder: ReadDecoder {
                reader,
                decoder,
                at_eof: false
            },
            buffer: Vec::with_capacity(32),
            current_frame: Frame::default(),
        }
    }
    
    fn init(mut self) -> Result<Self, DecodingError> {
        loop {
            match self.decoder.decode_next()? {
                Some(Decoded::BackgroundColor) => {}
                Some(Decoded::GlobalPalette) => {
                    break
                },
                Some(_) => {
                    // Unreachable since this loop exists after the global
                    // palette has been read.
                    unreachable!()
                },
                None => return Err(DecodingError::format(
                    "file does not contain any image data"
                ))
            }
        }
        Ok(self)
    }
    
    /// Returns the next frame info
    pub fn next_frame_info(&mut self) -> Result<Option<&Frame>, DecodingError> {
        loop {
            match self.decoder.decode_next()? {
                Some(Decoded::Frame(frame)) => {
                    self.current_frame = frame.clone();
                    break
                },
                Some(_) => (),
                None => return Ok(None)
                
            }
        }
        Ok(Some(&self.current_frame))
    }

    /// Reads the data of the current frame into a pre-allocated buffer.
    ///
    /// `Self::next_frame_info` needs to be called beforehand.
    /// The length of `buf` must be at least `Self::buffer_size`.
    /// Deinterlaces the result.
    pub fn read_into_buffer(&mut self) -> Result<(), DecodingError> {
        if self.current_frame.is_interlaced {
            let width = self.line_length();
            let height = self.current_frame.height as usize;
            for _row in (InterlaceIterator { len: height, next: 0, pass: 0 }) {
                if !self.fill_buffer(width)? {
                    return Err(DecodingError::format("image truncated"))
                }
            }
        } else {
            if !self.fill_buffer(self.buffer_size())? {
                return Err(DecodingError::format("image truncated"))
            }
        };
        Ok(())
    }

    /// Reads data of the current frame into a pre-allocated buffer until the buffer has been
    /// filled completely.
    ///
    /// `Self::next_frame_info` needs to be called beforehand. Returns `true` if the supplied
    /// buffer could be filled completely. Should not be called after `false` had been returned.
    pub fn fill_buffer(&mut self, mut wanted_buf_len: usize) -> Result<bool, DecodingError> {
        macro_rules! handle_data(
            ($data:expr) => {{
                let len = cmp::min(wanted_buf_len, $data.len());
                (len, 1)
            }}
        );
        let buf_len = self.buffer.len();
        if buf_len > 0 {
            let (len, channels) = handle_data!(&self.buffer);
            let _ = self.buffer.drain(..len);
            wanted_buf_len = wanted_buf_len.saturating_sub(len*channels);
            if wanted_buf_len == 0 {
                return Ok(true)
            }
        }
        loop {
            match self.decoder.decode_next()? {
                Some(Decoded::Data(data)) => {
                    let (len, channels) = handle_data!(data);
                    wanted_buf_len = wanted_buf_len.saturating_sub(len*channels);
                    if wanted_buf_len > 0 {
                        continue
                    } else if len < data.len() {
                        self.buffer.extend_from_slice(&data[len..]);
                    }
                    return Ok(true)
                },
                Some(_) => return Ok(false), // make sure that no important result is missed
                None => return Ok(false)
                
            }
        }
    }
    
    /// Output buffer size
    pub fn buffer_size(&self) -> usize {
        self.line_length() * self.current_frame.height as usize
    }
    
    /// Line length of the current frame
    pub fn line_length(&self) -> usize {
        self.current_frame.width as usize
    }
}

struct InterlaceIterator {
    len: usize,
    next: usize,
    pass: usize
}

impl iter::Iterator for InterlaceIterator {
    type Item = usize;

    fn next(&mut self) -> Option<Self::Item> {
        if self.len == 0 || self.pass > 3 {
            return None
        }
        let mut next = self.next + [8, 8, 4, 2][self.pass];
        while next >= self.len {
            next = [4, 2, 1, 0][self.pass];
            self.pass += 1;
        }
        mem::swap(&mut next, &mut self.next);
        Some(next)
    }
}

#[cfg(test)]
mod test {
    use super::InterlaceIterator;

    #[test]
    fn test_interlace_iterator() {
        for &(len, expect) in &[
            (0, &[][..]),
            (1, &[0][..]),
            (2, &[0, 1][..]),
            (3, &[0, 2, 1][..]),
            (4, &[0, 2, 1, 3][..]),
            (5, &[0, 4, 2, 1, 3][..]),
            (6, &[0, 4, 2, 1, 3, 5][..]),
            (7, &[0, 4, 2, 6, 1, 3, 5][..]),
            (8, &[0, 4, 2, 6, 1, 3, 5, 7][..]),
            (9, &[0, 8, 4, 2, 6, 1, 3, 5, 7][..]),
            (10, &[0, 8, 4, 2, 6, 1, 3, 5, 7, 9][..]),
            (11, &[0, 8, 4, 2, 6, 10, 1, 3, 5, 7, 9][..]),
            (12, &[0, 8, 4, 2, 6, 10, 1, 3, 5, 7, 9, 11][..]),
            (13, &[0, 8, 4, 12, 2, 6, 10, 1, 3, 5, 7, 9, 11][..]),
            (14, &[0, 8, 4, 12, 2, 6, 10, 1, 3, 5, 7, 9, 11, 13][..]),
            (15, &[0, 8, 4, 12, 2, 6, 10, 14, 1, 3, 5, 7, 9, 11, 13][..]),
            (16, &[0, 8, 4, 12, 2, 6, 10, 14, 1, 3, 5, 7, 9, 11, 13, 15][..]),
            (17, &[0, 8, 16, 4, 12, 2, 6, 10, 14, 1, 3, 5, 7, 9, 11, 13, 15][..]),
        ] {
            let iter = InterlaceIterator { len: len, next: 0, pass: 0 };
            let lines = iter.collect::<Vec<_>>();
            assert_eq!(lines, expect);
        }
    }
}
