use std::fmt;
use std::io::Error as IoError;

pub type Result<T, E=Error> = ::std::result::Result<T, E>;

/// Errors that can occur while decoding a JPEG image.
#[derive(Debug)]
pub enum Error {
    /// The image is not formatted properly. The string contains detailed information about the
    /// error.
    Format(&'static str),
    /// The image makes use of a JPEG feature not (currently) supported by this library.
    Unsupported,
    /// An I/O error occurred while decoding the image.
    Io(IoError),
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Self::Format(ref desc) => f.write_str(desc),
            Self::Unsupported => f.write_str("unsupported JPEG feature"),
            Self::Io(ref err) => err.fmt(f),
        }
    }
}

#[cfg(not(target_arch = "wasm32"))]
impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        match *self {
            Self::Io(ref err) => Some(err),
            _ => None,
        }
    }
}

impl From<IoError> for Error {
    fn from(err: IoError) -> Self {
        Self::Io(err)
    }
}
