use crate::framework::{auth, auth::AuthClient, endpoint::Endpoint, ApiResultTraits, Environment};
use async_trait::async_trait;
use serde::Serialize;
use surf::Result;

use serde_json::value::Value as JsonValue;

#[async_trait]
pub trait AsyncApiClient {
    async fn request<ResultType, QueryType, BodyType>(
        &self,
        endpoint: &(dyn Endpoint<ResultType, QueryType, BodyType> + Send + Sync),
    ) -> Result<ResultType>
    where
        ResultType: ApiResultTraits,
        QueryType: Serialize,
        BodyType: Serialize;
}

/// A Cloudflare API client that makes requests asynchronously.
pub struct Client {
    environment: Environment,
    credentials: auth::Credentials,
    client: surf::Client,
}

impl Client {
    pub fn new(credentials: auth::Credentials, environment: Environment) -> Result<Client> {
        let client = surf::client();
        Ok(Client {
            environment,
            credentials,
            client,
        })
    }
}

#[async_trait]
impl AsyncApiClient for Client {
    async fn request<ResultType, QueryType, BodyType>(
        &self,
        endpoint: &(dyn Endpoint<ResultType, QueryType, BodyType> + Send + Sync),
    ) -> Result<ResultType>
    where
        ResultType: ApiResultTraits,
        QueryType: Serialize,
        BodyType: Serialize,
    {
        let mut request =
            surf::RequestBuilder::new(endpoint.method(), endpoint.url(&self.environment));

        request = request.query(&endpoint.query())?;
        request = request.auth(&self.credentials);

        if let Some(body) = endpoint.body() {
            request = request.body(surf::Body::from_json(&body)?);
        }

        let req = request.build();
        log::debug!("Making {} request to {}", req.method(), req.url());
        req.iter()
            .for_each(|header| log::debug!("Request header {}:{}", header.0, header.1));

        let mut res = self.client.send(req).await?;
        let api_result = res.body_json::<ApiSuccess<ResultType>>().await?;

        Ok(api_result.result)
    }
}

#[derive(Deserialize, Debug, PartialEq)]
pub struct ApiSuccess<ResultType> {
    pub result: ResultType,
    pub result_info: Option<JsonValue>,
    pub messages: JsonValue,
    pub errors: Vec<JsonValue>,
}

/*
async fn map_api_response<ResultType: ApiResultTraits>(resp: surf::Response) -> Result<ResultType> {
    let status = resp.status();
    if status.is_success() {
        let api_result = resp.body_json::<ApiSuccess<ResultType>>().await;
        match api_result {
            Ok(res) => Ok(res),
            Err(e) => Err(ApiFailure::Invalid(e)),
        }
    } else {
        let api_result = res.body_json::<ApiErrors>().await;
        let errors = api_result.unwrap_or_default();
        Err(ApiFailure::Error(status, errors))
    }
}
*/
