#[macro_use]
mod util;

use chrono::{DateTime, Utc};
use cloudevents::event::{
    AttributesReader, EventBuilder, EventBuilderError, ExtensionValue, SpecVersion,
};
use cloudevents::EventBuilderV03;
use std::convert::TryInto;
use url::Url;

#[test]
fn build_event() {
    let id = "aaa";
    let source = "http://localhost:8080";
    let ty = "bbb";
    let subject = "francesco";
    let time: DateTime<Utc> = Utc::now();
    let extension_name = "ext";
    let extension_value = 10i64;
    let content_type = "application/json";
    let schema = Url::parse("http://localhost:8080/schema").unwrap();
    let data = serde_json::json!({
        "hello": "world"
    });

    let mut event = EventBuilderV03::new()
        .id(id)
        .source(source.clone())
        .ty(ty)
        .subject(subject)
        .time(time)
        .extension(extension_name, extension_value)
        .data_with_schema(content_type, schema.clone(), data.clone())
        .build()
        .unwrap();

    assert_eq!(SpecVersion::V03, event.specversion());
    assert_eq!(id, event.id());
    assert_eq!(source, event.source().clone());
    assert_eq!(ty, event.ty());
    assert_eq!(subject, event.subject().unwrap());
    assert_eq!(time, event.time().unwrap().clone());
    assert_eq!(
        ExtensionValue::from(extension_value),
        event.extension(extension_name).unwrap().clone()
    );
    assert_eq!(content_type, event.datacontenttype().unwrap());
    assert_eq!(schema, event.dataschema().unwrap().clone());

    let event_data: serde_json::Value = event.take_data().2.unwrap().try_into().unwrap();
    assert_eq!(data, event_data);
}

#[test]
fn source_valid_relative_url() {
    let res = EventBuilderV03::new()
        .id("id1")
        .source("/source") // relative URL
        .ty("type")
        .build();
    assert_match_pattern!(res, Ok(_));
}

#[test]
fn build_missing_id() {
    let res = EventBuilderV03::new()
        .source("http://localhost:8080")
        .build();
    assert_match_pattern!(
        res,
        Err(EventBuilderError::MissingRequiredAttribute {
            attribute_name: "id"
        })
    );
}

#[test]
fn source_invalid_url() {
    let res = EventBuilderV03::new().source("").build();
    assert_match_pattern!(
        res,
        Err(EventBuilderError::InvalidUriRefError {
            attribute_name: "source",
        })
    );
}

#[test]
fn default_builds() {
    let res = EventBuilderV03::default().build();
    assert_match_pattern!(res, Ok(_));
}
