use super::common;
use structopt::StructOpt;
use serde::{Serialize, Deserialize};

#[derive(Debug, StructOpt, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize)]
pub struct Task {
    /*
    https://developer.todoist.com/rest/v1/?shell#get-an-active-task
    Accepts a single ID, or multiple; chains GET requests in the case of multiple.
    */
    #[structopt(flatten)]
    pub id_args: common::OptionalIdArgs,

    /* View flags */
    #[structopt(long)]
    pub inbox: Option<bool>,
    #[structopt(long)]
    pub today: Option<bool>,
    #[structopt(long)]
    pub upcoming: Option<bool>,

    /** Todoist query format: https://todoist.com/help/articles/introduction-to-filters */
    #[structopt(short = "q", long = "fq")]
    pub filter_query: Option<String>,

    /* https://developer.todoist.com/rest/v1/?shell#tasks */
    #[structopt(flatten)]
    pub project_args: common::ProjectArgs,
    #[structopt(flatten)]
    pub label_args: common::LabelArgs,

    /* Subcommands: */
    #[structopt(subcommand)]
    pub command: Option<Command>,
}

impl crate::Runs<super::Todoist> for Task {
    fn run(app: &mut super::Todoist) {
        /*
        Even though Task::run is only going to be called by a function that has already performed
        this match, we're likely passing the same arguments to this function with the intention of
        passing some other data, but this time we can safely assume that only Task will match.
        */
        match &mut app.command {
            super::Command::Task(task) => {
                if let Some(command) = &task.command
                /* User passed a subcommand: */
                {
                    match command {
                        Command::Add(_) => commands::Add::run(task),
                        Command::Close(_) => commands::Close::run(task),
                        Command::Open(_) => commands::Open::run(task),
                        Command::Update(_) => commands::Update::run(task),
                        Command::Delete(_) => commands::Delete::run(task),
                        Command::Copy(_) => commands::Copy::run(task),
                    }
                } else
                /* User is making a task GET request: */
                {
                    // let mut reqwest = Reqwest::default();
                    // let mut is_due: Vec<Model> = Vec::new();
                    // let mut not_due: Vec<Model> = Vec::new();

                    // if let Some(list) = &task.id_args.id
                    // /* For ID(s) update the endpoint and fire away. */
                    // {
                    //     for id in list {
                    //         let mut model: Model = reqwest
                    //             .init(task.method(), &format!("{}/{}", task.endpoint(), id))
                    //             .add_auth(&app.auth_token)
                    //             .data()
                    //             .expect("error returned from `Todoist`");
                    //         model.compute_due_in();
                    //         if model.due_in.is_some() {
                    //             is_due.push(model);
                    //         } else {
                    //             not_due.push(model);
                    //         }
                    //     }
                    // } else
                    // /* If no ID, then build a query. */
                    // {
                    //     let mut query = Query::default();
                    //     if let Some(val) = &task.filter_query {
                    //         query.set_filter(Some(val.to_string()))
                    //     }
                    //     if let Some(val) = &task.project_args.project_id {
                    //         query.set_project_id(Some(*val))
                    //     }
                    //     if let Some(val) = &task.label_args.label_id {
                    //         query.set_label_id(Some(*val))
                    //     }

                    //     let list: Vec<Model> = reqwest
                    //         .init(task.method(), &task.endpoint())
                    //         .add_auth(&app.auth_token)
                    //         .add_query(&query)
                    //         .data()
                    //         .expect("error returned from `Todoist`");

                    //     for mut model in list {
                    //         model.compute_due_in();
                    //         if model.due_in.is_some() {
                    //             is_due.push(model);
                    //         } else {
                    //             not_due.push(model);
                    //         }
                    //     }
                    // }

                    // /* Sort by `due` `datetime` or `date`. */
                    // is_due.sort_by(|x, y| x.due_in_unix.cmp(&y.due_in_unix));
                    // for model in is_due {
                    //     println!("{}", model);
                    // }
                }
            }
            /* If another match occurs then weird things are happening so panic. */
            _ => {
                panic!("`Task::run` matched on a `Command` other than `Task`")
            }
        }
    }
}

#[derive(Debug, StructOpt, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize)]
pub enum Command {
    /** Add a new task */
    Add(commands::Add),
    /** Mark a task as complete; close a task */
    Close(commands::Close),
    /** Mark a task as incomplete; reopen a task */
    Open(commands::Open),
    /** Update an existing task */
    Update(commands::Update),
    /** Delete a task */
    Delete(commands::Delete),
    /** Duplicate a task; copy a task to a different project */
    Copy(commands::Copy),
}

mod commands;
