use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use structopt::StructOpt;

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct IdArgs {
    #[structopt(short, long)]
    pub id: Vec<usize>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct OptionalIdArgs {
    #[structopt(short, long)]
    pub id: Option<Vec<usize>>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct ProjectArgs {
    #[structopt(long = "pj", name = "project-id")]
    pub project_id: Option<usize>,
    #[structopt(long = "pjn", name = "project-name")]
    pub project_name: Option<String>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct LabelArgs {
    #[structopt(long = "lb", name = "label-id")]
    pub label_id: Option<usize>,
    #[structopt(long = "lbn", name = "label-name")]
    pub label_name: Option<String>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct ContentArgs {
    #[structopt(short, long)]
    pub content: Option<String>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct MandatoryContentArgs {
    /** Mandatory content */
    #[structopt(short, long)]
    pub content: String,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct DescriptionArgs {
    #[structopt(long)]
    pub desc: Option<String>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct NameArgs {
    #[structopt(short, long)]
    pub name: Option<String>,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct MandatoryNameArgs {
    /** Mandatory content */
    #[structopt(short, long)]
    pub name: String,
}

#[derive(
    Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize, JsonSchema,
)]
pub enum Priority {
    Low,
    Medium,
    High,
    Urgent,
}

impl Default for Priority {
    fn default() -> Self {
        Self::Low
    }
}

impl From<Priority> for u8 {
    fn from(prio: Priority) -> Self {
        match prio {
            Priority::Low => 1,
            Priority::Medium => 2,
            Priority::High => 3,
            Priority::Urgent => 4,
        }
    }
}

impl From<Priority> for String {
    fn from(prio: Priority) -> Self {
        match prio {
            Priority::Low => String::from("low"),
            Priority::Medium => String::from("medium"),
            Priority::High => String::from("high"),
            Priority::Urgent => String::from("urgent"),
        }
    }
}

impl std::str::FromStr for Priority {
    type Err = std::fmt::Error;
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        if s.contains("low") {
            Ok(Self::Low)
        } else if s.contains("medium") {
            Ok(Self::Medium)
        } else if s.contains("high") {
            Ok(Self::High)
        } else if s.contains("urgent") {
            Ok(Self::Urgent)
        } else {
            panic!("failed to parse &str as `Priority`")
        }
    }
}

impl std::fmt::Display for Priority {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Priority::Low => write!(f, "low"),
            Priority::Medium => write!(f, "medium"),
            Priority::High => write!(f, "high"),
            Priority::Urgent => write!(f, "urgent"),
        }
    }
}

#[derive(Debug, StructOpt, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize)]
pub struct PriorityArgs {
    /** low (1; default); medium (2); high (3); urgent (4) */
    #[structopt(long, default_value)]
    pub prio: Priority,
}

#[derive(
    Debug, StructOpt, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize,
)]
pub struct DueDateArgs {
    /** Due date as a string, e.g. every Wednesday */
    #[structopt(long = "dstring")]
    pub due_string: Option<String>,
    /** Format: [YYYY-MM-DD] */
    #[structopt(short = "d", long = "date")]
    pub due_date: Option<String>,
    /** Format: [HH:MM], valid examples: 7am, 7pm, 19:00, 1900, 19 */
    #[structopt(short = "t", long = "time")]
    pub due_time: Option<String>,
    #[structopt(long = "datetime")]
    pub due_date_time: Option<String>,
    /** Task recurrence flag */
    #[structopt(short, long)]
    pub recurring: bool,
}

#[derive(
    Debug, Default, Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize, JsonSchema,
)]
pub struct DueDateModel {
    /** Human defined date in arbitrary format */
    pub string: String,
    /** Date in format `YYYY-MM-DD` corrected to user's time zone */
    pub date: String,
    pub recurring: bool,
    /** Only returned if exact due time set (i.e. it's not a whole-day task), in UTC. */
    pub datetime: Option<String>,
    /**
    Only returned if exact due time set, user's time zone definition either in tzdata-compatible
    format ("Europe/Berlin") or as a string specifying east of UTC offset as "UTC±HH:MM"
    (i.e. "UTC-01:00").
    */
    pub timezone: Option<String>,
    /**
    In case DueDateTime.string is not in English, a two character language code can be provided
    */
    pub lang: Option<[char; 2]>,
}
