use crate::file_system::UseFileSystem;
use std::path::PathBuf;
use structopt::StructOpt;

#[derive(Debug, StructOpt, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Todoist {
    /** API token here: https://todoist.com/app/settings/integrations */
    #[structopt(long = "auth", global = true, default_value)]
    pub auth_token: String,

    /**
    Config file can be used instead of auth token, and will be used in auth_token isn't present
    */
    #[structopt(long, global = true)]
    pub config: Option<PathBuf>,

    #[structopt(subcommand)]
    pub command: Command,
}

const URL: &'static str = "https://api.todoist.com/rest/v1";

impl crate::Runs<Todoist> for Todoist {
    fn run(args: &mut Todoist) {
        /*
        If a token has been provided via CLI, use that and proceed, otherwise, we'll set it from
        config.
        */
        match &args.command {
            Command::Task(_) => task::Task::run(args),
            Command::Project(_) => todo!(),
            Command::Label(_) => todo!(),
        }
    }
}

#[derive(Debug, StructOpt, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub enum Command {
    /** Get or interact w/ tasks */
    Task(task::Task),
    /** Get or interact w/ projects; also lists tasks belonging to a project */
    Project(project::Project),
    /** Get or interact w/ labels; also lists tasks with given label(s) */
    Label(label::Label),
}

impl UseFileSystem for Todoist {
    fn data_dir(&self) -> PathBuf {
        let mut path = Self::base_dir();
        path.push("todoist");
        path
    }
    fn file_name(&self) -> PathBuf {
        "config.json".into()
    }
}

mod common;
mod display;
mod label;
mod project;
mod task;

pub struct Console {
    pub term: console::Term,
}

impl Default for Console {
    fn default() -> Self {
        Self {
            term: console::Term::stdout(),
        }
    }
}

impl Console {
    fn exit(&self) {
        self.term
            .write_line("Happy doing! 👋")
            .expect("failed to write exit line to term");
        std::process::exit(0)
    }
}
