pub trait UseFileSystem {
    /** `clogi` base dir. */
    fn base_dir() -> std::path::PathBuf {
        format!(
            "{}/{}",
            dirs::data_dir()
                .expect("no data dir?")
                .to_str()
                .expect("invalid path to data dir"),
            "clogi"
        )
        .into()
    }

    /* Store data here. */
    fn data_dir(&self) -> std::path::PathBuf;
    fn file_name(&self) -> std::path::PathBuf;

    /** Helper fn to make dirs recursively. Checks that the dir is within `base_dir` first. */
    fn make_dirs(&self) -> Result<(), std::io::Error> {
        if !self.data_dir().starts_with(Self::base_dir()) {
            return Err(std::io::Error::new(
                std::io::ErrorKind::Other,
                "attempted to make dir outside of base dir",
            ));
        }

        /* This recursive method shouldn't really _fail_, ever, but might behave weird. */
        std::fs::DirBuilder::new()
            .recursive(true)
            .create(self.data_dir())
    }
    /** Helper fn to stitch together `data_dir` and `file_name`. */
    fn full_path(&self) -> std::path::PathBuf {
        format!(
            "{}/{}",
            self.data_dir().to_str().unwrap(),
            self.file_name().to_str().unwrap()
        )
        .into()
    }
    /** Helper fn to read file. */
    fn read_to_string(&self) -> String {
        std::fs::read_to_string(self.full_path()).expect("failed to read file contents")
    }
    /** Helper fn to write a file. */
    fn write_file(&self, buf: &[u8]) -> Result<usize, std::io::Error> {
        if !self.data_dir().is_dir() {
            self.make_dirs().expect("failed to make dirs");
        }

        std::io::Write::write(
            &mut std::fs::OpenOptions::new()
                .read(true)
                .write(true)
                .truncate(true)
                .create(true)
                .open(self.full_path())
                .expect("failed to open file"),
            buf,
        )
    }
}
