use serde::{Deserialize, Serialize};
use structopt::StructOpt;

#[derive(Debug, StructOpt)]
pub(crate) struct DueDateArgs {
    #[structopt(long = "dstring")]
    pub due_string: Option<String>,
    #[structopt(short = "d", long = "date")]
    pub due_date: Option<String>,
    #[structopt(short = "t", long = "time")]
    pub due_time: Option<String>,
    #[structopt(long = "datetime")]
    pub due_date_time: Option<String>,
    #[structopt(short, long)]
    pub recurring: bool,
}

/// Assists with setting, and filtering on, due dates; does some pre-processing for my own flavour
/// of writing out due dates, but most of the heavy lifting is done by Todoist, as they already
/// support some degree of natural language parsing; NLP (natural-language programming)
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub(crate) struct DueDateTime {
    /// Human defined date in arbitrary format
    string: String,
    /// Date in format `YYYY-MM-DD` corrected to user's time zone
    date: String,
    recurring: bool,
    /// Only returned if exact due time set (i.e. it's not a whole-day task)
    date_time: Option<String>,
    /// Only returned if exact due time set, user's time zone definition either in
    /// tzdata-compatible format ("Europe/Berlin") or as a string specifying east of UTC offset as
    /// "UTC±HH:MM" (i.e. "UTC-01:00").
    time_zone: Option<String>,
    /// In case DueDateTime.string is not in English, a two character language code can be
    /// provided
    lang: Option<[char; 2]>,
}

impl Default for DueDateTime {
    fn default() -> Self {
        Self {
            string: "now".to_string(),
            date: "1993-30-10".to_string(),
            recurring: false,
            date_time: None,
            time_zone: None,
            lang: None,
        }
    }
}
