mod common_args;
mod due_date;
mod label;
mod project;
mod task;

use crate::Runs;
use structopt::StructOpt;

/**
The `Todoist` struct is a wrapper around subcommands, 'global' args can go here.

Examples:
- `clogi todoist --token abcd1234 task new --args xyz`
- `clogi todoist task 42` -> Get `task` with id of 42 (GET)
- `clogi todoist task list --project everyday` -> List all tasks under `everyday` project (GET)
- `clogi todoist task update 42 --args xyz` -> Update `task` 42, provide args + values for fields (PUT)
- `clogi todoist project new --name foo` -> Create a new `project` with name `foo` (POST)

_Note_: I'd advise aliasing `clogi todoist` to something else in your shell. I use `ct --token t`.

Typical command lifecycle looks like this:
1.  App parses arguments to determine that a `todoist` subcommand `task` has been issued.
2.  Construct method(s) to make a GET request to the `task` endpoint using `reqwest`
    1.  Usually constructs a client (`reqwest`), and a cache (`HashMap<K, V>`), to make the request
        and store data.
    2.  Parse and serialize user data - `serde` comes into play here.
3.  Build the request, adding arguments to body and/or token to header.
4.  Send the request, *asynchronously* or in a *blocking* fashion, `reqwest` can do both.
    -   Sometimes we'll need to chain a few requests together, and then filter the data before
        displaying to the user. In these instances, we'll also store the data locally so we're not
        doing this too often. There is a request limit after all.
5.  Display something meaningful to user.
*/
#[derive(Debug, StructOpt)]
pub struct Todoist {
    #[structopt(long, global = true)]
    pub(self) token: Option<String>,
    #[structopt(subcommand)]
    sub_command: SubCommand,
}

#[derive(Debug, StructOpt)]
#[structopt(name = "todoist", about = "A CLI for _individuals_ using Todoist")]
enum SubCommand {
    Task {
        #[structopt(subcommand)]
        args: task::Args,
    },
    Project {
        #[structopt(subcommand)]
        args: project::Args,
    },
    Label {
        #[structopt(subcommand)]
        args: label::Args,
    },
    // IGNORED: section; comment; collaborator
}

pub(self) const _BASE_URL: &str = "https://api.todoist.com/rest/v1";

impl Runs<Todoist> for Todoist {
    /// Entry point for `Todoist` app code.
    fn run(args: Todoist) {
        /*
        Optional token; some requests might require one and some might not, so we set it to an
        empty string by default; HEAD requests don't need a token.
        */
        let mut _token: String = String::from("");
        if let Some(arg) = args.token {
            _token = arg
        }

        /*
        Basically a "switch" on `SubCommand` enum variants, then run relevant code.

        Notice how this code is different to the code from `main.rs`:
        `Command::Todoist { todoist } => Todoist::run(todoist)`

        We haven't got code for `Task` that looks like:
        `Command::Task { task } => Task::run(task)`

        Instead, we have:
        `SubCommand::Task { args } => task::run_task(args)`

        `Todoist` is a struct, so an _application_ - it houses all `Todoist` code, so it implements
        a trait that means it's adhering to the rules of being an _application_, or a _command_.

        Our `SubCommand` are enums since different rules apply, you don't want to have a struct for
        each `SubCommand` - enums enable us to choose which subcommand to rule most effectively.
        */
        match args.sub_command {
            SubCommand::Task { args } => task::run_task(args),
            SubCommand::Project { args } => project::run_project(args),
            SubCommand::Label { args } => label::run_label(args),
        }
    }
}
