use std::{
    sync::{Arc, RwLock},
    time::{Duration, SystemTime, UNIX_EPOCH},
};

/// Milliseconds since epoch at 2015-05-15T00:00:00 UTC.
///
/// This is the date of the [Rust 1.0 announcement][rust-announcement].
///
/// [rust-announcement]: https://blog.rust-lang.org/2015/05/15/Rust-1.0.html
const DEFAULT_MILLIS_SINCE_EPOCH: u64 = 1431648000000;

/// Fake clock.
#[derive(Debug, Clone)]
pub struct FakeClock {
    clock: Arc<RwLock<SystemTime>>,
}

/// Create a new [`FakeClock`] with the time `2015-05-15T00:00:00 UTC`.
impl Default for FakeClock {
    fn default() -> Self {
        Self::from(Duration::from_millis(DEFAULT_MILLIS_SINCE_EPOCH))
    }
}

impl FakeClock {
    pub fn now(&self) -> SystemTime {
        *self.clock.read().unwrap()
    }

    /// Advance the clock by `duration`.
    pub fn advance(&self, duration: Duration) {
        let mut clock = self.clock.write().unwrap();
        *clock += duration;
    }

    /// Set the clock to time `t`.
    pub fn set(&self, t: SystemTime) {
        let mut clock = self.clock.write().unwrap();
        *clock = t;
    }
}

/// Create a new [`FakeClock`] at time since [`UNIX_EPOCH`].
impl From<Duration> for FakeClock {
    fn from(value: Duration) -> Self {
        Self::from(UNIX_EPOCH + value)
    }
}

/// Create a new [`FakeClock`] at [`SystemTime`].
impl From<SystemTime> for FakeClock {
    fn from(t: SystemTime) -> Self {
        Self { clock: Arc::new(RwLock::new(t)) }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn from_system_time() {
        let now = SystemTime::now();
        let clock = FakeClock::from(now);
        assert_eq!(clock.now(), now);
    }

    #[test]
    fn from_duration() {
        let now = SystemTime::now();
        let since_epoch = now.duration_since(UNIX_EPOCH).unwrap();
        let clock = FakeClock::from(since_epoch);
        assert_eq!(clock.now(), now);
    }

    #[test]
    fn advance_secs() {
        let now = SystemTime::now();
        let clock = FakeClock::from(now);
        clock.advance(Duration::from_secs(1));
        assert_eq!(clock.now(), now + Duration::from_secs(1));
    }

    #[test]
    fn advance_nanos() {
        let now = SystemTime::now();
        let clock = FakeClock::from(now);
        clock.advance(Duration::from_nanos(1));
        assert_eq!(clock.now(), now + Duration::from_nanos(1));
    }

    #[test]
    fn set_time() {
        let clock = FakeClock::default();
        let t = SystemTime::now();
        assert_ne!(clock.now(), t);

        clock.set(t);
        assert_eq!(clock.now(), t);
    }
}
