use std::path::PathBuf;

use git2::{Oid, Repository};
use url::Url;

use crate::config::{Config, Dependency};

pub fn fetch_dependencies(config: &Config) -> Vec<PathBuf> {
    let mut res = Vec::with_capacity(config.dependency.len());

    for dependency in &config.dependency {
        fetch_dependency(dependency, &mut res);
    }

    res
}

/// Fetches a dependency and stores it in the cache
fn fetch_dependency(dependency: &Dependency, paths: &mut Vec<PathBuf>) {
    let url = Url::parse(&dependency.url).expect("Could not parse dependency url");
    let mut dir = dirs::home_dir().expect("Could not get home dir of user");
    dir.push(".cache/cllm/dependencies/");
    let repo_name = url
        .path_segments()
        .expect("Could not get target path from dependency url")
        .last()
        .expect("Coult not get target path from dependency url");

    dir.push(repo_name);
    dir.set_extension("");
    dir.push(&dependency.revision);

    let repo = match Repository::open(&dir) {
        Ok(_) => {
            eprintln!(
                "Repository {} with revision {} found in cache",
                repo_name, dependency.revision
            );
            for path in &dependency.paths {
                let mut new_path = dir.clone();
                new_path.push(path);
                paths.push(new_path);
            }
            return;
        }
        Err(_) => {
            eprintln!(
                "Cloning {} with revision {}",
                repo_name, dependency.revision
            );
            match Repository::clone_recurse(&dependency.url, &dir) {
                Ok(repo) => {
                    for path in &dependency.paths {
                        let mut new_path = dir.clone();
                        new_path.push(path);
                        paths.push(new_path);
                    }
                    repo
                }
                Err(_) => panic!("Could not clone dependecy git module"),
            }
        }
    };

    match repo.set_head_detached(
        Oid::from_str(&dependency.revision).expect("Revision could not be parsed as a commit"),
    ) {
        Ok(_) => (),
        Err(_) => panic!("Coult not fetch revision"),
    }
}
