use serde::{Deserialize, Serialize};
use std::{
    fs::{read_to_string, write},
    path::PathBuf,
};

#[derive(Debug, Deserialize, Serialize)]
pub struct Config {
    pub name: String,
    pub main_module: String,
    pub paths: Vec<PathBuf>,
    pub dependency: Vec<Dependency>,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Dependency {
    pub url: String,
    pub revision: String,
    pub paths: Vec<PathBuf>,
}

const CONFIG_FILE: &str = "cllm.toml";

fn default_config() -> Config {
    let stdenv = Dependency {
        url: String::from("https://gitlab.science.ru.nl/clean-compiler-and-rts/stdenv.git"),
        revision: String::from("ENTER_COMMIT_HERE"),
        paths: vec![PathBuf::from("StdEnv 64 Changed Files"), PathBuf::from(".")],
    };

    Config {
        name: String::from("default-name"),
        main_module: String::from("Main"),
        paths: vec![PathBuf::from("src")],
        dependency: vec![stdenv],
    }
}

/// Create new template config file
pub fn write_new_config() {
    let config = default_config();
    let config = toml::to_string(&config)
        .expect("Could not convert default config to string. Please report this error.");
    write(CONFIG_FILE, config).expect(&format!("Could not write to {}", CONFIG_FILE));
    eprintln!("Wrote new {}", CONFIG_FILE);
}

/// Parses the CONFIG_FILE file in the current directory
pub fn parse_config() -> Config {
    let config: String = read_to_string(CONFIG_FILE).expect(&format!(
        "Could not find {} in current directory.",
        CONFIG_FILE
    ));
    let config: Config =
        toml::from_str(&config).expect(&format!("Could not parse {}", CONFIG_FILE));
    config
}
