use std::any::Any;
use std::str::FromStr;

use crate::entities::errors::CliArgError;

pub type ArgParseResult = Result<Box<dyn Any>, CliArgError>;

pub struct Param {
    pub name: String,
    pub parser: Box<dyn Fn(String) -> ArgParseResult>,
    pub _description: Option<String>,
    pub _short: Option<String>,
    pub _long: Option<String>,
    pub _default: Option<String>,
    pub _optional: bool,
    pub _cmd_enum: Vec<(String, String)>,
}

pub struct UseCaseMeta {
    pub description: &'static str,
    pub params: Vec<Param>,
}

pub struct RawUseCaseRequest {
    pub params: Vec<Option<Box<dyn Any>>>,
    pub rest_args: Vec<String>,
    pub cmd: String,
    meta: UseCaseMeta,
}

impl Param {
    pub fn new<T: ToString>(name: T, parser: Box<dyn Fn(String) -> ArgParseResult>) -> Param {
        Param {
            name: name.to_string(),
            parser,
            _description: None,
            _short: None,
            _long: None,
            _default: None,
            _optional: false,
            _cmd_enum: Vec::new(),
        }
    }

    pub fn description<T: ToString>(mut self, data: T) -> Param {
        self._description = Some(data.to_string());
        self
    }

    pub fn short<T: ToString>(mut self, data: T) -> Param {
        self._short = Some(data.to_string());
        self
    }

    pub fn long<T: ToString>(mut self, data: T) -> Param {
        self._long = Some(data.to_string());
        self
    }

    pub fn default<T: ToString>(mut self, data: T) -> Param {
        self._default = Some(data.to_string());
        self
    }

    pub fn optional(mut self, flag: bool) -> Param {
        self._optional = flag;
        self
    }
}

impl UseCaseMeta {
    pub fn new(description: &'static str) -> UseCaseMeta {
        UseCaseMeta {
            description,
            params: Vec::new(),
        }
    }

    pub fn param(mut self, param: Param) -> UseCaseMeta {
        self.params.push(param);
        self
    }

    pub fn make_enum<A: ToString, B: ToString>(mut self, data: Vec<(A, B)>) -> UseCaseMeta {
        let variants = data
            .iter()
            .map(|(key, _)| key.to_string())
            .collect::<Vec<_>>();
        let mut param = Param::new("command", enum_validator(&variants));
        param._cmd_enum = data
            .into_iter()
            .map(|(a, b)| (a.to_string(), b.to_string()))
            .collect();
        self.params.push(param);
        self
    }

    pub fn is_enum(&self) -> bool {
        !self.params.is_empty() && !self.params[0]._cmd_enum.is_empty()
    }
}

impl UseCaseRequest {
    pub fn new_uc_request(
        meta: UseCaseMeta,
        cmd: String,
        params: Vec<Option<Box<dyn Any>>>,
    ) -> UseCaseRequest {
        UseCaseRequest {
            params,
            meta,
            rest_args: vec![],
            cmd,
        }
    }

    pub fn new_enum_value(
        meta: UseCaseMeta,
        cmd: String,
        command: Box<dyn Any>,
        rest_args: Vec<String>,
    ) -> UseCaseRequest {
        UseCaseRequest {
            meta,
            params: vec![Some(command)],
            cmd,
            rest_args,
        }
    }

    pub fn get_opt<T: Clone + 'static>(&self, key: &str) -> Option<T> {
        for i in 0..self.meta.params.len() {
            if self.meta.params[i].name == key {
                return self.open_param(key, i);
            }
        }
        None
    }

    pub fn get<T: Clone + 'static>(&self, key: &str) -> T {
        match self.get_opt(key) {
            Some(val) => val,
            _ => panic!("UseCaseRequest[{}]: not set", key),
        }
    }

    fn open_param<T: Clone + 'static>(&self, key: &str, index: usize) -> Option<T> {
        let any = self.params[index].as_ref()?;
        let opt: Option<&T> = any.downcast_ref();
        if let Some(value) = opt {
            Some((*value).clone())
        } else {
            panic!("UseCaseRequest[{}]: type missmatch", key)
        }
    }
}
