macro_rules! impl_error {
    ($struct_name:ident) => {
        impl std::fmt::Display for $struct_name {
            fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                write!(f, "{:?}", self)
            }
        }
        impl std::error::Error for $struct_name {}
    };
}

#[derive(Debug, Eq, PartialEq)]
pub enum CliArgError {
    NotFound(String),
    InvalidVariant(String),
    InvalidOption(String),
    ExpectedValue(String),
    UnexpectedParam(String),
}

#[derive(Debug)]
pub enum UseCaseError {
    InvalidRequest(String),
    RuntimeError(String),
}

impl UseCaseError {
    pub fn make_inv_req<T: ToString>(msg: T) -> UseCaseError {
        UseCaseError::InvalidRequest(msg.to_string())
    }
    pub fn raise_inv_req<T: ToString, V>(msg: T) -> Result<V, UseCaseError> {
        Err(UseCaseError::make_inv_req(msg))
    }
    pub fn make_runtime<T: ToString>(msg: T) -> UseCaseError {
        UseCaseError::RuntimeError(msg.to_string())
    }
    pub fn raise_runtime<T: ToString, V>(msg: T) -> Result<V, UseCaseError> {
        Err(UseCaseError::make_runtime(msg))
    }
}

#[derive(Debug)]
pub enum ConfigError {
    InvalidVar(String),
    VarNotSet(String, String),
}

impl std::fmt::Display for ConfigError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::InvalidVar(name) => write!(f, "Invalid var \"{}\"", name),
            Self::VarNotSet(name, description) => {
                write!(f, "Var not set \"{}\" ({})", name, description)
            }
        }
    }
}

impl std::error::Error for ConfigError {}

impl_error!(CliArgError);
impl_error!(UseCaseError);
