use std::env;
use std::env::VarError;

use crate::entities::errors::ConfigError;

pub struct ConfigFieldSchema<Config> {
    pub name: &'static str,
    pub description: &'static str,
    pub default_value: Option<String>,
    pub setter: Box<dyn Fn(&mut Config, &str) -> Result<(), ConfigError>>,
    pub getter: Box<dyn Fn(&Config) -> String>,
}

pub struct ConfigFieldState {
    pub name: &'static str,
    pub description: &'static str,
    pub default_value: Option<String>,
    pub current_value: String,
}

fn set_default<Config>(
    config: &mut Config,
    field: &ConfigFieldSchema<Config>,
) -> Result<(), ConfigError> {
    if let Some(ref default) = field.default_value {
        (field.setter)(config, default)
    } else {
        Err(ConfigError::VarNotSet(
            field.name.into(),
            field.description.into(),
        ))
    }
}

pub fn build_config<Config>(
    config: &mut Config,
    fields: &[ConfigFieldSchema<Config>],
) -> Result<(), ConfigError> {
    for field in fields {
        let name = field.name;
        match env::var(name) {
            Ok(val) if val.is_empty() => set_default(config, field)?,
            Ok(val) => (field.setter)(config, &val)?,
            Err(VarError::NotPresent) => set_default(config, field)?,
            _ => return Err(ConfigError::InvalidVar(name.into())),
        };
    }
    Ok(())
}

pub fn render_config<Config>(
    config: &Config,
    fields: &[ConfigFieldSchema<Config>],
) -> Vec<ConfigFieldState> {
    fields
        .iter()
        .map(|field| ConfigFieldState {
            name: field.name,
            description: field.description,
            default_value: field.default_value.clone(),
            current_value: (field.getter)(config),
        })
        .collect::<Vec<_>>()
}

impl<Config> ConfigFieldSchema<Config> {
    pub fn new<
        T: ToString,
        E,
        Setter: 'static + Fn(&mut Config, &str) -> Result<(), E>,
        Getter: 'static + Fn(&Config) -> &T,
    >(
        name: &'static str,
        description: &'static str,
        default: Option<T>,
        setter: Setter,
        getter: Getter,
    ) -> ConfigFieldSchema<Config> {
        let setter = Box::new(move |config: &mut Config, text: &str| {
            if setter(config, text).is_err() {
                Err(ConfigError::InvalidVar(name.to_string()))
            } else {
                Ok(())
            }
        });
        let getter = Box::new(move |config: &Config| getter(config).to_string());
        ConfigFieldSchema {
            name,
            description,
            default_value: default.map(|x| x.to_string()),
            setter,
            getter,
        }
    }
}
