use std::any::Any;
use std::collections::HashMap;

use crate::entities::cli::use_case::{CliUseCaseMeta, RawUseCaseRequest};
use crate::entities::errors::CliArgError;
use crate::entities::errors::CliArgError::InvalidOption;
use crate::entities::errors::UseCaseError;
use crate::proto::cli::ICliUseCase;

struct Schema {
    nokeys: Vec<usize>,
    shorts: HashMap<String, usize>,
    longs: HashMap<String, usize>,
    next_nokey_index: usize,
}

pub struct Parser {
    meta: CliUseCaseMeta,
}

impl Schema {
    fn gen(meta: &CliUseCaseMeta) -> Schema {
        let mut nokeys = Vec::new();
        let mut shorts = HashMap::new();
        let mut longs = HashMap::new();
        for (i, param) in meta.params.iter().enumerate() {
            if param._short.is_none() && param._long.is_none() {
                nokeys.push(i);
                continue;
            }
            if let Some(ref val) = param._short {
                shorts.insert(val.clone(), i);
            }
            if let Some(ref val) = param._long {
                longs.insert(val.clone(), i);
            }
        }
        Schema {
            nokeys,
            shorts,
            longs,
            next_nokey_index: 0,
        }
    }

    fn get_key_index(&self, token: &str) -> Result<usize, CliArgError> {
        let chars = token.chars().collect::<Vec<_>>();
        if chars.len() < 2 {
            return Err(CliArgError::InvalidOption(token.to_string()));
        }
        let opt_index = if chars[1] == '-' {
            let key = chars[2..].iter().collect::<String>();
            self.longs.get(&key).copied()
        } else {
            let key = chars[1..].iter().collect::<String>();
            self.shorts.get(&key).copied()
        };
        match opt_index {
            Some(index) => Ok(index),
            _ => Err(InvalidOption(token.to_string())),
        }
    }

    fn get_nokey_index(&mut self, token: &str) -> Result<usize, CliArgError> {
        if self.next_nokey_index >= self.nokeys.len() {
            return Err(CliArgError::UnexpectedParam(token.to_string()));
        }
        let index = self.nokeys[self.next_nokey_index];
        self.next_nokey_index += 1;
        Ok(index)
    }
}

impl Parser {
    pub fn call_use_case<UC: ICliUseCase<Response = ()>, A: ToString>(
        mut use_case: UC,
        args: &[A],
    ) -> Result<(), UseCaseError> {
        let meta = use_case.get_meta();
        let parser = Parser::new(meta);
        let args = args.iter().map(|x| x.to_string()).collect::<Vec<_>>();
        let request = parser
            .parse_request(String::new(), &args)
            .expect("invalid args");
        use_case.execute(use_case.validate_request(request))
    }

    pub fn new(meta: CliUseCaseMeta) -> Parser {
        Parser { meta }
    }

    pub fn parse_request(
        self,
        cmd: String,
        args: &[String],
    ) -> Result<RawUseCaseRequest, CliArgError> {
        if self.meta.is_enum() {
            self.parse_enum_request(cmd, args)
        } else {
            self.parse_uc_request(cmd, args)
        }
    }

    fn parse_enum_request(
        self,
        cmd: String,
        args: &[String],
    ) -> Result<RawUseCaseRequest, CliArgError> {
        let schema = &self.meta.params[0];
        if args.is_empty() {
            let param_name = schema.name.clone();
            return Err(CliArgError::NotFound(param_name));
        }
        let value = schema.validator.validate(args[0].clone())?;
        let response = RawUseCaseRequest::new_enum_value(self.meta, cmd, value, args[1..].to_vec());
        Ok(response)
    }

    fn parse_uc_request(
        self,
        cmd: String,
        args: &[String],
    ) -> Result<RawUseCaseRequest, CliArgError> {
        let mut schema = Schema::gen(&self.meta);
        let mut parsed = (0..self.meta.params.len())
            .map(|_| None)
            .collect::<Vec<Option<Box<dyn Any>>>>();
        let mut i = 0;
        while i < args.len() {
            let token = &args[i];
            if self.is_key(token) {
                if i + 1 == args.len() {
                    return Err(CliArgError::ExpectedValue(token.to_string()));
                }
                let index = schema.get_key_index(token)?;
                let value = self.parse_value(index, &args[i + 1])?;
                parsed[index] = Some(value);
                i += 2;
            } else {
                let index = schema.get_nokey_index(token)?;
                let value = self.parse_value(index, token)?;
                parsed[index] = Some(value);
                i += 1;
            }
        }
        self.validate_parsed(&mut parsed)?;
        Ok(RawUseCaseRequest::new_request(self.meta, cmd, parsed))
    }

    fn is_key(&self, token: &str) -> bool {
        token.starts_with('-')
    }

    fn parse_value(&self, index: usize, token: &str) -> Result<Box<dyn Any>, CliArgError> {
        let param = &self.meta.params[index];
        param.validator.validate(token.to_string())
    }

    fn validate_parsed(&self, parsed: &mut Vec<Option<Box<dyn Any>>>) -> Result<(), CliArgError> {
        for (i, schema) in self.meta.params.iter().enumerate() {
            if parsed[i].is_some() {
                continue;
            }
            if let Some(ref dflt) = schema._default {
                let value = schema.validator.validate(dflt.clone()).unwrap();
                parsed[i] = Some(value);
                continue;
            }
            if !schema._optional {
                return Err(CliArgError::NotFound(schema.name.clone()));
            }
        }
        Ok(())
    }
}
