use crate::entities::cli::use_case::{CliUseCaseMeta, Param};
use std::fmt;
use std::fmt::Write;

fn get_key(param: &Param) -> String {
    if let Some(ref value) = param._short {
        return format!("-{} {}", value, param.name);
    }
    if let Some(ref value) = param._long {
        return format!("--{} {}", value, param.name);
    }
    param.name.clone()
}

fn make_short_line(result: &mut String, command: &str, meta: &CliUseCaseMeta) -> fmt::Result {
    write!(result, "{} ", command)?;
    for param in meta.params.iter() {
        let key = get_key(param);
        if param._optional || param._default.is_some() {
            write!(result, "[{}] ", key)?;
        } else {
            write!(result, "{} ", key)?;
        }
    }
    writeln!(result)
}

fn make_enum_help(result: &mut String, param: &Param) -> fmt::Result {
    writeln!(result, "{} variants:", param.name)?;
    for (key, desc) in param._cmd_enum.iter() {
        writeln!(result, "  {} - {}", key, desc)?;
    }
    Ok(())
}

fn make_param_help(result: &mut String, param: &Param) -> fmt::Result {
    write!(result, "  ")?;
    if let Some(ref short) = param._short {
        write!(result, "-{} ", short)?;
    }
    if let Some(ref long) = param._long {
        write!(result, "--{} ", long)?;
    }
    if param._optional {
        result.push('[');
    }
    result.push_str(&param.name);
    if let Some(ref value) = param._default {
        write!(result, "={}", value)?;
    }
    if param._optional {
        result.push(']');
    }
    if let Some(ref description) = param._description {
        writeln!(result, ": {}", description)?;
    }
    Ok(())
}

pub fn generate_help(command: &str, meta: CliUseCaseMeta) -> String {
    let mut result = format!("{} : {}\n", command, meta.description);
    if meta.params.is_empty() {
        return result;
    }
    make_short_line(&mut result, command, &meta).unwrap();
    if meta.is_enum() {
        make_enum_help(&mut result, &meta.params[0]).unwrap()
    } else {
        writeln!(result, "Arguments:").unwrap();
        for param in meta.params {
            make_param_help(&mut result, &param).unwrap()
        }
    }
    result
}
