import pytest
from utils import shell

def test_no_command():
    output = shell("test_proj", expect_error=True)
    assert output == ['Error NotFound("command")']

def test_global_help():
    output = shell("test_proj -h")
    assert output == [
        'test_proj : test app',
        'test_proj command ',
        'command variants:',
        '  config - show config',
        '  p - commands with params',
    ]

@pytest.mark.parametrize('command', ["test_proj config", "PROP_B= test_proj config"])
def test_get_conf_err(command):
    output = shell(command, expect_error=True)
    assert output == ['Error Var not set "PROP_B" (str value)']

def test_get_conf_ok():
    output = shell("PROP_B=abc test_proj config")
    assert output == [
        "PROP_A (usize value): '0' DEFAULT 'Some(\"0\")'",
        "PROP_B (str value): 'abc' DEFAULT 'None'",
    ]

def test_get_conf_set_var():
    output = shell("PROP_A=3 PROP_B=1 test_proj config")
    assert output == [
        "PROP_A (usize value): '3' DEFAULT 'Some(\"0\")'",
        "PROP_B (str value): '1' DEFAULT 'None'",
    ]

def test_get_conf_set_var_err():
    output = shell("PROP_A=abc PROP_B=1 test_proj config", expect_error=True)
    assert output == ['Error Invalid var "PROP_A"']

def test_branch():
    output = shell("test_proj p", expect_error=True)
    assert output == ['Error NotFound("command")']

def test_branch_h():
    output = shell("test_proj p -h")
    assert output == [
        'test_proj p : commands with params',
        'test_proj p command ',
        'command variants:',
        '  hello - say hello',
        '  math - do math',
    ]

def test_hello_err():
    output = shell("test_proj p hello", expect_error=True)
    assert output == ['Error NotFound("_name_")']

@pytest.mark.parametrize(
    "cmd,out",
    [
        ("-n Alex", ["Hello Alex!"]),
        ("--name K", ["Hello K!"]),
        ("-t y -n K", ["Hello K!", "My name is Mr. Robot"]),
        ("-n K -t 1", ["Hello K!", "My name is Mr. Robot"]),
        ("-n K -t T", ["Hello K!", "My name is Mr. Robot"]),
        ("-n K -t f", ["Hello K!"]),
        ("-n K -t N abc", ["Hello K!", "> abc"]),
        ("abc -n K -t y", ["Hello K!", "My name is Mr. Robot", "> abc"]),
    ],
)
def test_hello(cmd, out):
    output = shell(f"test_proj p hello {cmd}")
    assert output == out

def test_math_ok():
    output = shell("test_proj p math 1 + 2")
    assert output == ['1 + 2 = 3']

def test_math_bad_op():
    output = shell("test_proj p math 1 x 2", expect_error=True)
    assert output == ['Error InvalidVariant("x")']

def test_math_bad_nan():
    output = shell("test_proj p math 1 + r", expect_error=True)
    assert output == ['Error InvalidVariant("invalid param: r. Expected type: f64")']
