use crate::entities::cli::use_case::ArgParseResult;
use crate::entities::errors::CliArgError;
use crate::proto::cli::IParamValidator;
use std::any::{type_name, Any};
use std::marker::PhantomData;
use std::str::FromStr;

pub struct EnumValidator {
    variants: Vec<String>,
    return_index: bool,
}

#[derive(Default)]
pub struct BoolValidator {}
#[derive(Default)]
pub struct AsIsValidator {}
#[derive(Default)]
pub struct FromStrValidator<T> {
    phantom: PhantomData<T>,
}

impl EnumValidator {
    pub fn ret_values<T: ToString>(variants: &[T]) -> EnumValidator {
        let variants = variants
            .iter()
            .map(|x| x.to_string())
            .collect::<Vec<String>>();
        EnumValidator {
            variants,
            return_index: false,
        }
    }

    pub fn ret_indexes<T: ToString>(variants: &[T]) -> EnumValidator {
        let mut validator = Self::ret_values(variants);
        validator.return_index = true;
        validator
    }
}

impl IParamValidator for EnumValidator {
    fn validate(&self, param: String) -> ArgParseResult {
        for i in 0..self.variants.len() {
            if self.variants[i] == param {
                let dyn_val: Box<dyn Any> = if self.return_index {
                    Box::new(i)
                } else {
                    Box::new(self.variants[i].clone())
                };
                return Ok(dyn_val);
            }
        }
        Err(CliArgError::InvalidVariant(param))
    }
}

impl IParamValidator for BoolValidator {
    fn validate(&self, param: String) -> ArgParseResult {
        const VARIANTS: &[(&str, bool)] = &[
            ("1", true),
            ("y", true),
            ("t", true),
            ("0", false),
            ("n", false),
            ("f", false),
        ];
        let param = param.to_lowercase();
        for (var, val) in VARIANTS.iter() {
            if *var == param {
                return Ok(Box::new(*val));
            }
        }
        Err(CliArgError::InvalidVariant(param))
    }
}

impl IParamValidator for AsIsValidator {
    fn validate(&self, param: String) -> ArgParseResult {
        Ok(Box::new(param))
    }
}

impl<T: FromStr + 'static> IParamValidator for FromStrValidator<T> {
    fn validate(&self, param: String) -> ArgParseResult {
        if let Ok(val) = T::from_str(&param) {
            return Ok(Box::new(val));
        };
        let msg = format!(
            "invalid param: {}. Expected type: {}",
            param,
            type_name::<T>(),
        );
        Err(CliArgError::InvalidVariant(msg))
    }
}
