use crate::cli::helper::generate_help;
use crate::cli::param_parser::Parser;
use crate::cli::use_case_wrapper::UseCaseWrapper;
use std::env;
use std::fmt::Display;
use std::process::exit;

use crate::entities::cli::use_case::{CliUseCaseMeta, RawUseCaseRequest, COMMAND_PARAM};
use crate::entities::errors::{CliArgError, ConfigError, UseCaseError};
use crate::proto::cli::{ICliUseCase, ICliUseCaseWrapped};
use crate::proto::di::IUseCase;

type Factory<T> = Box<dyn FnOnce() -> Result<T, ConfigError>>;

pub struct Runner {
    use_case_factory: Factory<Box<dyn ICliUseCaseWrapped>>,
    description: &'static str,
    command: &'static str,
}

pub struct RunnerTree {
    runners: Vec<Runner>,
    description: &'static str,
}

fn error_exit<E: Display>(err: E) -> ! {
    eprintln!("Error {}", err);
    exit(1)
}

fn make_wrapped_factory<
    UC: 'static + ICliUseCase<Response = ()>,
    F: 'static + FnOnce() -> Result<UC, ConfigError>,
>(
    original: F,
) -> Factory<Box<dyn ICliUseCaseWrapped>> {
    Box::new(move || {
        let use_case: UC = original()?;
        let wrapped: UseCaseWrapper<UC> = UseCaseWrapper::new(use_case);
        let boxed: Box<dyn ICliUseCaseWrapped> = Box::new(wrapped);
        Ok(boxed)
    })
}

impl Runner {
    pub fn new_leaf<
        UC: 'static + ICliUseCase<Response = ()>,
        F: 'static + FnOnce() -> Result<UC, ConfigError>,
    >(
        factory: F,
        description: &'static str,
        command: &'static str,
    ) -> Runner {
        Runner {
            use_case_factory: make_wrapped_factory(factory),
            description,
            command,
        }
    }

    pub fn new_main<
        UC: 'static + ICliUseCase<Response = ()>,
        F: 'static + FnOnce() -> Result<UC, ConfigError>,
    >(
        factory: F,
        description: &'static str,
    ) -> Runner {
        Runner::new_leaf(factory, description, "")
    }

    pub fn run(self) {
        let args: Vec<String> = env::args().collect();
        self.run_with_args(&args[0], &args[1..])
    }

    pub fn run_with_args(self, cmd: &str, args: &[String]) {
        let mut use_case = match (self.use_case_factory)() {
            Ok(val) => val,
            Err(err) => error_exit(err),
        };
        let request = match Runner::parse_args(&*use_case, cmd, args) {
            Ok(val) => val,
            Err(err) => error_exit(err),
        };
        if let Err(err) = use_case.execute(request) {
            error_exit(err)
        }
    }

    fn show_help(use_case: &dyn ICliUseCaseWrapped, cmd: &str) {
        let meta = use_case.get_meta();
        println!("{}", generate_help(cmd, meta));
    }

    fn parse_args(
        use_case: &dyn ICliUseCaseWrapped,
        cmd: &str,
        args: &[String],
    ) -> Result<RawUseCaseRequest, CliArgError> {
        if !args.is_empty() && (args[0] == "-h" || args[0] == "--help") {
            Runner::show_help(use_case, cmd);
            exit(0);
        }
        let meta = use_case.get_meta();
        Parser::new(meta).parse_request(cmd.to_string(), args)
    }
}

impl RunnerTree {
    pub fn fold(runners: Vec<Runner>, description: &'static str) -> RunnerTree {
        RunnerTree {
            description,
            runners,
        }
    }

    #[allow(dead_code)]
    pub fn into_runner(self, command: &'static str) -> Runner {
        let description = self.description;
        Runner::new_leaf(|| Ok(self), description, command)
    }

    pub fn into_main_runner(self) -> Runner {
        let description = self.description;
        Runner::new_main(|| Ok(self), description)
    }

    // fn into_use_case_factory(self) -> UseCaseFactory {
    //     let use_case: Box<dyn IUseCase> = Box::new(RunnerTree {
    //         runners: self.runners,
    //         description: self.description,
    //     });
    //     Box::new(move || Ok(use_case))
    // }
}

pub struct RunnerTreeRequest {
    full_command: String,
    index: usize,
    rest_args: Vec<String>,
}

impl IUseCase for RunnerTree {
    type Request = RunnerTreeRequest;
    type Response = ();

    fn execute(&mut self, request: Self::Request) -> Result<Self::Response, UseCaseError> {
        let runner = self.runners.remove(request.index);
        runner.run_with_args(&request.full_command, &request.rest_args);
        Ok(())
    }
}

impl ICliUseCase for RunnerTree {
    fn get_meta(&self) -> CliUseCaseMeta {
        let commands = self
            .runners
            .iter()
            .map(|runner| (runner.command, runner.description))
            .collect::<Vec<(&'static str, &'static str)>>();
        CliUseCaseMeta::new(self.description).make_enum(commands)
    }

    fn validate_request(&self, raw: RawUseCaseRequest) -> Self::Request {
        let index = raw.get(COMMAND_PARAM);
        let (ref cmd, _) = self.get_meta().params[0]._cmd_enum[index];
        RunnerTreeRequest {
            index,
            full_command: format!("{} {}", raw.cmd, cmd),
            rest_args: raw.rest_args,
        }
    }
}
