use std::env;
use std::env::VarError;

use crate::entities::errors::ConfigError;

// const MIN_DIFF_DIST_NAME: &str = "RZIMG_MIN_DIFF_DIST";
// const MAX_DIFF_PERCENT_NAME: &str = "RZIMG_MAX_DIFF_PERCENT";
// const MAX_DIFF_PERCENT_SMALL_NAME: &str = "RZIMG_MAX_DIFF_PERCENT_SMALL";
// const MAX_THUMBNAIL_SIZE_NAME: &str = "RZIMG_THUMBNAIL_SIZE";
// const MAX_THREADS_NAME: &str = "RZIMG_MAX_THREADS";
//
// const MIN_DIFF_DIST_DEFAULT: isize = 5;
// const MAX_DIFF_PERCENT_DEFAULT: f64 = 0.001;
// const MAX_DIFF_PERCENT_SMALL_DEFAULT: f64 = 0.01;
// const MAX_THUMBNAIL_SIZE_DEFAULT: u32 = 50;
// const MAX_THREADS_DEFAULT: usize = 4;

// pub struct Config {
//     pub min_diff_dist: isize,
//     pub max_diff_percent_full: f64,
//     pub max_diff_percent_small: f64,
//     pub max_thumbnail_size: u32,
//     pub max_threads: usize,
// }

pub struct ConfigFieldSchema<Config> {
    pub name: &'static str,
    pub description: &'static str,
    pub default_value: String,
    // pub current_value: String,
    pub setter: Box<dyn Fn(&mut Config, &str) -> Result<(), ConfigError>>,
    pub getter: Box<dyn Fn(&Config) -> String>,
}

pub struct ConfigFieldState {
    pub name: &'static str,
    pub description: &'static str,
    pub default_value: String,
    pub current_value: String,
}

pub fn build_config<Config>(
    config: &mut Config,
    fields: &[ConfigFieldSchema<Config>],
) -> Result<(), ConfigError> {
    for field in fields {
        let name = field.name;
        match env::var(name) {
            Ok(val) => (field.setter)(config, &val)?,
            Err(VarError::NotPresent) => (field.setter)(config, &field.default_value)?,
            _ => return Err(ConfigError::InvalidVar(name.to_string())),
        };
    }
    Ok(())
}

pub fn render_config<Config>(
    config: &Config,
    fields: &[ConfigFieldSchema<Config>],
) -> Vec<ConfigFieldState> {
    fields
        .iter()
        .map(|field| ConfigFieldState {
            name: field.name,
            description: field.description,
            default_value: field.default_value.clone(),
            current_value: (field.getter)(config),
        })
        .collect::<Vec<_>>()
}

impl<Config> ConfigFieldSchema<Config> {
    pub fn new<
        T: ToString,
        E,
        Setter: 'static + Fn(&mut Config, &str) -> Result<(), E>,
        Getter: 'static + Fn(&Config) -> &T,
    >(
        name: &'static str,
        description: &'static str,
        default: T,
        setter: Setter,
        getter: Getter,
    ) -> ConfigFieldSchema<Config> {
        let setter = Box::new(move |config: &mut Config, text: &str| {
            if setter(config, text).is_err() {
                Err(ConfigError::InvalidVar(name.to_string()))
            } else {
                Ok(())
            }
        });
        let getter = Box::new(move |config: &Config| getter(config).to_string());
        ConfigFieldSchema {
            name,
            description,
            default_value: default.to_string(),
            setter,
            getter,
        }
    }
}

// impl Config {
//     pub fn scheme(&self) -> Vec<ConfigField> {
//         vec![
//             ConfigField::new(
//                 MIN_DIFF_DIST_NAME,
//                 "Min pixels distance interpreted as error",
//                 MIN_DIFF_DIST_DEFAULT,
//                 self.min_diff_dist,
//             ),
//             ConfigField::new(
//                 MAX_DIFF_PERCENT_NAME,
//                 "Max different pixels available to duplicates[0 .. 1]",
//                 MAX_DIFF_PERCENT_DEFAULT,
//                 self.max_diff_percent_full,
//             ),
//             ConfigField::new(
//                 MAX_DIFF_PERCENT_SMALL_NAME,
//                 "Max different pixels available for thumbnails[0 .. 1]",
//                 MAX_DIFF_PERCENT_SMALL_DEFAULT,
//                 self.max_diff_percent_small,
//             ),
//             ConfigField::new(
//                 MAX_THUMBNAIL_SIZE_NAME,
//                 "Max thumbnail side(in pixels) used for comparing",
//                 MAX_THUMBNAIL_SIZE_DEFAULT,
//                 self.max_thumbnail_size,
//             ),
//             ConfigField::new(
//                 MAX_THREADS_NAME,
//                 "Max available threads which will be used in parallel operations",
//                 MAX_THREADS_DEFAULT,
//                 self.max_threads,
//             ),
//         ]
//     }
//
//     pub fn build() -> Result<Config, ConfigError> {
//         let res = Config {
//             min_diff_dist: Config::get_var(MIN_DIFF_DIST_NAME, MIN_DIFF_DIST_DEFAULT)?,
//             max_diff_percent_full: Config::get_var(
//                 MAX_DIFF_PERCENT_NAME,
//                 MAX_DIFF_PERCENT_DEFAULT,
//             )?,
//             max_thumbnail_size: Config::get_var(
//                 MAX_THUMBNAIL_SIZE_NAME,
//                 MAX_THUMBNAIL_SIZE_DEFAULT,
//             )?,
//             max_diff_percent_small: Config::get_var(
//                 MAX_DIFF_PERCENT_SMALL_NAME,
//                 MAX_DIFF_PERCENT_SMALL_DEFAULT,
//             )?,
//             max_threads: Config::get_var(MAX_THREADS_NAME, MAX_THREADS_DEFAULT)?,
//         };
//         Ok(res)
//     }
//
//     fn get_var<T: FromStr>(name: &str, default: T) -> Result<T, ConfigError> {
//         let val = match env::var(name) {
//             Ok(val) => val,
//             Err(VarError::NotPresent) => return Ok(default),
//             _ => return Err(ConfigError::InvalidVar(name.to_string())),
//         };
//         if val.is_empty() {
//             return Ok(default);
//         }
//         T::from_str(&val).map_err(|_| ConfigError::InvalidVar(name.to_string()))
//     }
// }
