use crate::campaign::Campaign;

use std::collections::HashMap;

use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize};
use std::fmt::{Display, Formatter, Result};

#[derive(Clone, Copy, Debug, Deserialize, Eq, Hash, PartialEq, Serialize)]
pub enum Feature {
    #[serde(rename = "lang")]
    Lang,

    #[serde(rename = "ua")]
    Ua
}

impl Display for Feature {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result {
        match self {
            Feature::Lang => write!(f, "lang"),
            Feature::Ua => write!(f, "ua")
        }
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
pub struct Consent {

    pub granted: bool,

    pub start: DateTime<Utc>,

    pub end: DateTime<Utc>,
}

impl Consent {

    pub fn new(granted: bool) -> Consent {
        Consent { granted, start: Utc::now(), end: Utc::now() }
    }
}

#[derive(Debug, Deserialize, Serialize)]
pub struct FeatureConsent {

    pub feature: Feature,

    pub granted: bool,

    pub start: DateTime<Utc>,

    pub end: DateTime<Utc>,
}

impl FeatureConsent {

    pub fn new(feature: Feature, consent: &Consent) -> FeatureConsent {
        FeatureConsent { feature, granted: consent.granted, start: consent.start, end: consent.end }
    }
}

#[derive(Debug, Deserialize, Serialize)]
pub struct CampaignConsent {

    pub campaign_id: String,

    pub granted: bool,

    pub start: DateTime<Utc>,

    pub end: DateTime<Utc>,
}

impl CampaignConsent {

    pub fn new(campaign_id: String, consent: &Consent) -> CampaignConsent {
        CampaignConsent {
            campaign_id, granted: consent.granted, start: consent.start, end: consent.end }
    }
}

/// This struct keeps track of all granted or denied consents of a user.
///
/// There are two different types of consents:
/// - Consents for common features like if we're allowed to evaluate the locale or a user agent.
/// - Consents per measurement campaign.
///
/// The time of the consent is recorded along with it's state: If it was actually granted or denied.
///
/// Consents for common features are given indefinitely, since they are only ever recorded along
/// with running campaigns.
///
/// Consents for campaigns only last for a certain amount of days.
#[derive(Debug, Deserialize, PartialEq, Serialize)]
pub struct Consents {

    pub features: HashMap<Feature, Consent>,

    pub campaigns: HashMap<String, Consent>,
}

impl Consents {

    /// User consents to evaluate a `Feature`.
    pub fn grant_feature(&mut self, feature: Feature) -> FeatureConsent {
        // Don't overwrite original grant timestamp.
        if !self.features.contains_key(&feature) || !self.features[&feature].granted {
            self.features.insert(feature, Consent::new(true));
        }

        let consent = &self.features[&feature];

        FeatureConsent::new(feature, consent)
    }

    /// User denies consent to evaluate a `Feature`.
    pub fn deny(&mut self, feature: Feature) -> FeatureConsent {
        // Don't overwrite original deny timestamp.
        if !self.features.contains_key(&feature) || self.features[&feature].granted {
            self.features.insert(feature, Consent::new(false));
        }

        let consent = &self.features[&feature];

        FeatureConsent::new(feature, consent)
    }

    /// Returns if consent to a `Feature` was given.
    pub fn has_been_granted(&self, feature: Feature) -> bool {
        if !self.features.contains_key(&feature) {
            return false
        }

        self.features[&feature].granted
    }

    /// User consents to run a specific campaign.
    ///
    /// # Arguments
    /// * campaign_id: The campaign ID.
    /// * campaign: The campaign.
    pub fn grant_campaign(&mut self, campaign_id: &str, campaign: &Campaign) -> CampaignConsent {
        let period = campaign.next_total_measurement_period();

        match period {
            Some(period) => {
                // Always overwrite, since this might be a refreshed consent for a new period.
                self.campaigns.insert(campaign_id.to_string(), Consent { granted: true, start: period.0, end: period.1 });
            }
            None => {
                // Consent is technically granted, but has no effect, as start and end
                // will be set the same.
                self.campaigns.insert(campaign_id.to_string(), Consent::new(true));
            }
        }

        let consent = &self.campaigns[campaign_id];

        CampaignConsent::new(campaign_id.to_string(), consent)
    }

    /// User denies consent to run a specific campaign.
    pub fn deny_campaign(&mut self, campaign_id: &str) -> CampaignConsent {
        // Don't overwrite original deny timestamp.
        if !self.campaigns.contains_key(campaign_id) || self.campaigns[campaign_id].granted {
            self.campaigns.insert(campaign_id.to_string(), Consent::new(false));
        }

        let consent = &self.campaigns[campaign_id];

        CampaignConsent::new(campaign_id.to_string(), consent)
    }

    /// Returns if consent to run a campaign was given and is valid for the given period.
    pub fn is_campaign_granted(&self, campaign_id: &str, start: DateTime<Utc>, end: DateTime<Utc>) -> bool {

        // Do we have a consent at all? No: -> Discard.
        if let Some(consent) = self.get_consent(campaign_id) {
            // Is the start of the period inside the consent period? No: -> Discard.
            if start < consent.start {
                return false
            }

            // Is the end of the period inside the consent period? No: -> Discard.
            if end > consent.end {
                return false
            }

            // Hey! A valid consent.
            return true
        }

        false
    }

    /// Returns if consent to run a campaign was given and is now valid.
    pub fn is_campaign_currently_granted(&self, campaign_id: &str) -> bool {
        self.is_campaign_granted(campaign_id, Utc::now(), Utc::now())
    }

    fn get_consent(&self, campaign_id: &str) -> Option<&Consent> {

        // Do we have a consent at all? No: -> Discard.
        if !self.campaigns.contains_key(campaign_id) {
            return None
        }

        let consent = &self.campaigns[campaign_id];

        // Is it actually granted? No: -> Discard.
        if !consent.granted {
            return None
        }

        return Some(consent)
    }
}