use std::env;
use std::fs::{self, File};
use std::collections::{HashMap, BTreeMap};
use std::io::{prelude::*, BufReader, Read};

use crate::generators::{self, changes::{Changes, TOMLEdition}};
use crate::types::{MainConfig, MainConfigFile, LinkConfig};
use crate::console::load_session;
use crate::test::run_tests;

use clarity_repl::{repl};

use clap::Clap;
use toml;

#[derive(Clap)]
#[clap(version = "1.0")]
struct Opts {
    #[clap(subcommand)]
    command: Command,
}

#[derive(Clap)]
enum Command {
    /// New subcommand
    #[clap(name = "new")]
    New(GenerateProject),
    /// Contract subcommand
    #[clap(name = "contract")]
    Contract(Contract),
    /// Console subcommand
    #[clap(name = "console")]
    Console(Console),
    /// Test subcommand
    #[clap(name = "test")]
    Test(Test),
    /// Check subcommand
    #[clap(name = "check")]
    Check(Check),
}

#[derive(Clap)]
enum Contract {
    /// New contract subcommand
    #[clap(name = "new")]
    NewContract(NewContract),
    /// Import contract subcommand
    #[clap(name = "link")]
    LinkContract(LinkContract),
    /// Fork contract subcommand
    #[clap(name = "fork")]
    ForkContract(ForkContract),
}

#[derive(Clap)]
struct GenerateProject {
    /// Project's name
    pub name: String,
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
}

#[derive(Clap)]
struct NewContract {
    /// Contract's name
    pub name: String,
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
}

#[derive(Clap)]
struct LinkContract {
    /// Contract id
    pub contract_id: String,
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
}

#[derive(Clap, Debug)]
struct ForkContract {
    /// Contract id
    pub contract_id: String,
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
    // /// Fork contract and all its dependencies
    // #[clap(short = 'r')]
    // pub recursive: bool,
}

#[derive(Clap)]
struct Console {
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
}

#[derive(Clap)]
struct Test {
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
    pub files: Vec<String>,
}

#[derive(Clap)]
struct Check {
    /// Print debug info
    #[clap(short = 'd')]
    pub debug: bool,
}

pub fn main() {
    let opts: Opts = Opts::parse();

    let current_path = {
        let current_dir = env::current_dir().expect("Unable to read current directory");
        current_dir.to_str().unwrap().to_owned()
    };

    match opts.command {
        Command::New(project_opts) => {
            let changes = generators::get_changes_for_new_project(current_path, project_opts.name);
            execute_changes(changes);
        }
        Command::Contract(subcommand) => match subcommand {
            Contract::NewContract(new_contract) => {
                let changes =
                    generators::get_changes_for_new_contract(current_path, new_contract.name, None, true, vec![]);
                execute_changes(changes);
            }
            Contract::LinkContract(link_contract) => {
                let path = format!("{}/Clarinet.toml", current_path);

                let change = TOMLEdition {
                    comment: format!("Indexing link {} in Clarinet.toml", link_contract.contract_id),
                    path,
                    contracts_to_add: HashMap::new(),
                    links_to_add: vec![LinkConfig {
                        contract_id: link_contract.contract_id.clone(),
                    }],
                };
                execute_changes(vec![Changes::EditTOML(change)]);
            }
            Contract::ForkContract(fork_contract) => {
                let path = format!("{}/Clarinet.toml", current_path);

                println!("Resolving {} and its dependencies...", fork_contract.contract_id);

                let settings = repl::SessionSettings::default();
                let mut session = repl::Session::new(settings);

                let res = session.resolve_link(&repl::settings::InitialLink {
                    contract_id: fork_contract.contract_id.clone(),
                    stacks_node_addr: None,
                    cache: None,
                });
                let contracts = res.unwrap();
                let mut changes = vec![];
                for (contract_id, code, deps) in contracts.into_iter() {
                    let components: Vec<&str> = contract_id.split('.').collect();
                    let contract_name = components.last().unwrap();

                    if &contract_id == &fork_contract.contract_id {
                        let mut change_set =
                            generators::get_changes_for_new_contract(current_path.clone(), contract_name.to_string(), Some(code), false, vec![]);
                        changes.append(&mut change_set);

                        for dep in deps.iter() {
                            let mut change_set =
                                generators::get_changes_for_new_link(path.clone(), dep.clone(), None);
                            changes.append(&mut change_set);
                        }
                    }
                }
                execute_changes(changes);
            }

        },
        Command::Console(_) => {
            let start_repl = true;
            load_session(start_repl).expect("Unable to start REPL");
        },
        Command::Check(_) => {
            let start_repl = false;
            let res = load_session(start_repl);
            if let Err(e) = res {
                println!("{}", e);
                return;
            }
        },
        Command::Test(test) => {
            let start_repl = false;
            let res = load_session(start_repl);
            if let Err(e) = res {
                println!("{}", e);
                return;
            }
            run_tests(test.files);
        }
    };
}
  
fn execute_changes(changes: Vec<Changes>) {
    for mut change in changes.into_iter() {
        match change {
            Changes::AddFile(options) => {
                println!("{}", options.comment);
                let mut file = File::create(options.path.clone()).expect("Unable to create file");
                file.write_all(options.content.as_bytes())
                    .expect("Unable to write file");
            }
            Changes::AddDirectory(options) => {
                println!("{}", options.comment);
                fs::create_dir_all(options.path.clone()).expect("Unable to create directory");
            }
            Changes::EditTOML(ref mut options) => {
                let file = File::open(options.path.clone()).unwrap();
                let mut config_file_reader = BufReader::new(file);
                let mut config_file = vec![];
                config_file_reader.read_to_end(&mut config_file).unwrap();
                let config_file: MainConfigFile = toml::from_slice(&config_file[..]).unwrap();
                let mut config: MainConfig = MainConfig::from_config_file(config_file);
                let mut dirty = false;
                println!("BEFORE: {:?}", config);

                let mut links = match config.links.take() {
                    Some(links) => links,
                    None => vec![],
                };
                for link in options.links_to_add.drain(..) {
                    if links.contains(&link) {
                        links.push(link);
                        dirty = true;
                    }
                }
                config.links = Some(links);

                let mut contracts = match config.contracts.take() {
                    Some(contracts) => contracts,
                    None => BTreeMap::new(),
                };
                for (contract_name, contract_config) in options.contracts_to_add.iter() {
                    let res = contracts.insert(contract_name.clone(), contract_config.clone());
                    if res.is_none() {
                        dirty = true;
                    }
                }
                config.contracts = Some(contracts);

                println!("AFTER: {:?}", config);

                if dirty {
                    let toml = toml::to_string(&config).unwrap();
                    let mut file = File::create(options.path.clone()).unwrap();
                    file.write_all(&toml.as_bytes()).unwrap();    
                } 
                println!("{}", options.comment);
            }
        }
    }
}
