# Tutorial

*Jump to [builder tutorial](../tutorial_builder/README.md)*

1. [Quick Start](#quick-start)
2. [Configuring the Parser](#configuring-the-parser)
3. [Adding Arguments](#adding-arguments)
    1. [Flags](#flags)
    2. [Options](#options)
    3. [Positionals](#positionals)
    4. [Subcommands](#subcommands)
    5. [Defaults](#defaults)
4. Validation
    1. [Enumerated values](#enumerated-values)
    2. [Validated values](#validated-values)
    3. [Argument Relations](#relations-relations)
    4. [Custom Validation](#custom-validation)
5. [Tips](#tips)
6. [Contributing](#contributing)

## Quick Start

You can create an application declaratively with a `struct` and some
attributes.  **This requires enabling the `derive` feature flag.**

[Example:](01_quick.rs)
```bash
$ 01_quick_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    01_quick_derive[EXE] [OPTIONS] [NAME] [SUBCOMMAND]

ARGS:
    <NAME>    Optional name to operate on

OPTIONS:
    -c, --config <FILE>    Sets a custom config file
    -d, --debug            Turn debugging information on
    -h, --help             Print help information
    -V, --version          Print version information

SUBCOMMANDS:
    help    Print this message or the help of the given subcommand(s)
    test    does testing things
```

By default, the program does nothing:
```bash
$ 01_quick_derive
Debug mode is off
```

But you can mix and match the various features
```bash
$ 01_quick_derive -dd test
Debug mode is on
Not printing testing lists...
```

## Configuring the Parser

You use the `App` the start building a parser.

[Example:](02_apps.rs)
```bash
$ 02_apps_derive --help
MyApp 1.0
Kevin K. <kbknapp@gmail.com>
Does awesome things

USAGE:
    02_apps_derive[EXE] --two <TWO> --one <ONE>

OPTIONS:
    -h, --help         Print help information
        --one <ONE>    
        --two <TWO>    
    -V, --version      Print version information
$ 02_apps_derive --version
MyApp 1.0
```

You can use `app_from_crate!()` to fill these fields in from your `Cargo.toml` file.

[Example:](02_crate.rs)
```bash
$ 02_crate_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    02_crate_derive[EXE] --two <TWO> --one <ONE>

OPTIONS:
    -h, --help         Print help information
        --one <ONE>    
        --two <TWO>    
    -V, --version      Print version information
$ 02_crate_derive --version
clap [..]
```

You can use `AppSettings` to change the application level behavior of clap. You
can apply the setting to the top level command (`app.setting()`) or to it and
all subcommands (`app.global_setting()`).

[Example:](02_app_settings.rs)
```bash
$ 02_app_settings_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    02_app_settings_derive[EXE] --two <TWO> --one <ONE>

OPTIONS:
        --two <TWO>    
        --one <ONE>    
    -h, --help         Print help information
    -V, --version      Print version information
$ 02_app_settings_derive --one -1 --one -3 --two 10
two: "10"
one: "-3"
```

## Adding Arguments

### Flags

Flags are switches that can be on/off:

[Example:](03_01_flag_bool.rs)
```bash
$ 03_01_flag_bool_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_01_flag_bool_derive[EXE] [OPTIONS]

OPTIONS:
    -h, --help       Print help information
    -v, --verbose    
    -V, --version    Print version information
$ 03_01_flag_bool_derive
verbose: false
$ 03_01_flag_bool_derive --verbose
verbose: true
$ 03_01_flag_bool_derive --verbose --verbose
? failed
error: The argument '--verbose' was provided more than once, but cannot be used multiple times

USAGE:
    03_01_flag_bool_derive[EXE] [OPTIONS]

For more information try --help
```

Or counted.

[Example:](03_01_flag_count.rs)
```bash
$ 03_01_flag_count_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_01_flag_count_derive[EXE] [OPTIONS]

OPTIONS:
    -h, --help       Print help information
    -v, --verbose    
    -V, --version    Print version information
$ 03_01_flag_count_derive
verbose: 0
$ 03_01_flag_count_derive --verbose
verbose: 1
$ 03_01_flag_count_derive --verbose --verbose
verbose: 2
```

### Options

Flags can also accept a value.

[Example:](03_02_option.rs)
```bash
$ 03_02_option_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_02_option_derive[EXE] [OPTIONS]

OPTIONS:
    -h, --help           Print help information
    -n, --name <NAME>    
    -V, --version        Print version information
$ 03_02_option_derive
name: None
$ 03_02_option_derive --name bob
name: Some("bob")
$ 03_02_option_derive --name=bob
name: Some("bob")
$ 03_02_option_derive -n bob
name: Some("bob")
$ 03_02_option_derive -n=bob
name: Some("bob")
$ 03_02_option_derive -nbob
name: Some("bob")
```

### Positionals

Or you can have users specify values by their position on the command-line:

[Example:](03_03_positional.rs)
```bash
$ 03_03_positional_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_03_positional_derive[EXE] [NAME]

ARGS:
    <NAME>    

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information
$ 03_03_positional_derive
name: None
$ 03_03_positional_derive bob
name: Some("bob")
```

### Subcommands

Subcommands are defined as `App`s that get added via `App::subcommand`. Each
instance of a Subcommand can have its own version, author(s), Args, and even its own
subcommands.

[Example:](03_04_subcommands.rs)
```bash
$ 03_04_subcommands_derive
? failed
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_04_subcommands_derive[EXE] <SUBCOMMAND>

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information

SUBCOMMANDS:
    add     Adds files to myapp
    help    Print this message or the help of the given subcommand(s)
$ 03_04_subcommands_derive help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_04_subcommands_derive[EXE] <SUBCOMMAND>

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information

SUBCOMMANDS:
    add     Adds files to myapp
    help    Print this message or the help of the given subcommand(s)
$ 03_04_subcommands_derive help add
03_04_subcommands_derive[EXE]-add [..]
Adds files to myapp

USAGE:
    03_04_subcommands_derive[EXE] add [NAME]

ARGS:
    <NAME>    

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information
$ 03_04_subcommands_derive add bob
'myapp add' was used, name is: Some("bob")
```

Because we set `AppSettings::PropagateVersion`:
```bash
$ 03_04_subcommands_derive --version
clap [..]
$ 03_04_subcommands_derive add --version
03_04_subcommands_derive[EXE]-add [..]
```

### Defaults

We've previously showed that arguments can be `required` or optional.  When
optional, you work with a `Option` and can `unwrap_or`.  Alternatively, you can
set `Arg::default_value`.

[Example:](03_05_default_values.rs)
```bash
$ 03_05_default_values_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    03_05_default_values_derive[EXE] [NAME]

ARGS:
    <NAME>    [default: alice]

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information
$ 03_05_default_values_derive
name: "alice"
$ 03_05_default_values_derive bob
name: "bob"
```

## Validation

### Enumerated values

If you have arguments of specific values you want to test for, you can derive
`ArgEnum`.

This allows you specify the valid values for that argument. If the user does not use one of
those specific values, they will receive a graceful exit with error message informing them
of the mistake, and what the possible valid values are

[Example:](04_01_enum.rs)
```bash
$ 04_01_enum_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    04_01_enum_derive[EXE] <MODE>

ARGS:
    <MODE>    What mode to run the program in [possible values: fast, slow]

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information
$ 04_01_enum_derive fast
Hare
$ 04_01_enum_derive slow
Tortoise
$ 04_01_enum_derive medium
? failed
error: "medium" isn't a valid value for '<MODE>'
	[possible values: fast, slow]

USAGE:
    04_01_enum_derive[EXE] <MODE>

For more information try --help
```

### Validated values

More generally, you can validate and parse into any data type.

[Example:](04_02_validate.rs)
```bash
$ 04_02_validate_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    04_02_validate_derive[EXE] <PORT>

ARGS:
    <PORT>    Network port to use

OPTIONS:
    -h, --help       Print help information
    -V, --version    Print version information
$ 04_02_validate_derive 22
PORT = 22
$ 04_02_validate_derive foobar
? failed
error: Invalid value for '<PORT>': invalid digit found in string

For more information try --help
```

### Argument Relations

You can declare dependencies or conflicts between `Arg`s or even `ArgGroup`s.  

`ArgGroup`s  make it easier to declare relations instead of having to list each
individually, or when you want a rule to apply "any but not all" arguments.

Perhaps the most common use of `ArgGroup`s is to require one and *only* one argument to be
present out of a given set. Imagine that you had multiple arguments, and you want one of them to
be required, but making all of them required isn't feasible because perhaps they conflict with
each other.

[Example:](04_03_relations.rs)
```bash
$ 04_03_relations_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    04_03_relations_derive[EXE] [OPTIONS] <--set-ver <VER>|--major|--minor|--patch> [INPUT_FILE]

ARGS:
    <INPUT_FILE>    some regular input

OPTIONS:
    -c <CONFIG>                
    -h, --help                 Print help information
        --major                auto inc major
        --minor                auto inc minor
        --patch                auto inc patch
        --set-ver <VER>        set version manually
        --spec-in <SPEC_IN>    some special input argument
    -V, --version              Print version information
$ 04_03_relations_derive
? failed
error: The following required arguments were not provided:
    <--set-ver <VER>|--major|--minor|--patch>

USAGE:
    04_03_relations_derive[EXE] [OPTIONS] <--set-ver <VER>|--major|--minor|--patch> [INPUT_FILE]

For more information try --help
$ 04_03_relations_derive --major
Version: 2.2.3
$ 04_03_relations_derive --major --minor
? failed
error: The argument '--major' cannot be used with '--minor'

USAGE:
    04_03_relations_derive[EXE] <--set-ver <VER>|--major|--minor|--patch>

For more information try --help
$ 04_03_relations_derive --major -c config.toml
? failed
error: The following required arguments were not provided:
    <INPUT_FILE|--spec-in <SPEC_IN>>

USAGE:
    04_03_relations_derive[EXE] -c <CONFIG> <--set-ver <VER>|--major|--minor|--patch> <INPUT_FILE|--spec-in <SPEC_IN>>

For more information try --help
$ 04_03_relations_derive --major -c config.toml --spec-in input.txt
Version: 2.2.3
Doing work using input input.txt and config config.toml
```

### Custom Validation

As a last resort, you can create custom errors with the basics of clap's formatting.

[Example:](04_04_custom.rs)
```bash
$ 04_04_custom_derive --help
clap [..]
A simple to use, efficient, and full-featured Command Line Argument Parser

USAGE:
    04_04_custom_derive[EXE] [OPTIONS] [INPUT_FILE]

ARGS:
    <INPUT_FILE>    some regular input

OPTIONS:
    -c <CONFIG>                
    -h, --help                 Print help information
        --major                auto inc major
        --minor                auto inc minor
        --patch                auto inc patch
        --set-ver <VER>        set version manually
        --spec-in <SPEC_IN>    some special input argument
    -V, --version              Print version information
$ 04_04_custom_derive
? failed
error: Cam only modify one version field

USAGE:
    clap [OPTIONS] [INPUT_FILE]

For more information try --help
$ 04_04_custom_derive --major
Version: 2.2.3
$ 04_04_custom_derive --major --minor
? failed
error: Cam only modify one version field

USAGE:
    clap [OPTIONS] [INPUT_FILE]

For more information try --help
$ 04_04_custom_derive --major -c config.toml
? failed
Version: 2.2.3
error: INPUT_FILE or --spec-in is required when using --config

USAGE:
    clap [OPTIONS] [INPUT_FILE]

For more information try --help
$ 04_04_custom_derive --major -c config.toml --spec-in input.txt
Version: 2.2.3
Doing work using input input.txt and config config.toml
```

## Tips

- Proactively check for bad `App` configurations by calling `App::debug_assert` ([example](05_01_assert.rs))

## Contributing

New example code:
- Building: They must be added to [Cargo.toml](../../Cargo.toml) with the appropriate `required-features`.
- Testing: Ensure there is a markdown file with [trycmd](https://docs.rs/trycmd) syntax (generally they'll go in here).

See also the general [CONTRIBUTING](../../CONTRIBUTING.md).
