use serde::de::{Deserialize, Deserializer, Error, Unexpected, Visitor};
use std::fmt::{self, Debug, Display};

#[derive(Copy, Clone, Default, Eq, PartialEq, Ord, PartialOrd, Hash)]
pub struct Id {
    id: u64,
}

impl Id {
    pub const NULL: Id = Id { id: 0 };
}

impl Display for Id {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        write!(formatter, "0x{:x}", self.id)
    }
}

impl Debug for Id {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        write!(formatter, "Id({})", self)
    }
}

impl<'de> Deserialize<'de> for Id {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct IdVisitor;

        impl<'de> Visitor<'de> for IdVisitor {
            type Value = Id;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("clang syntax tree node id")
            }

            fn visit_str<E>(self, string: &str) -> Result<Self::Value, E>
            where
                E: Error,
            {
                string
                    .strip_prefix("0x")
                    .and_then(|hex| u64::from_str_radix(hex, 16).ok())
                    .map(|id| Id { id })
                    .ok_or_else(|| E::invalid_value(Unexpected::Str(string), &self))
            }
        }

        deserializer.deserialize_str(IdVisitor)
    }
}
