use std::fmt;
use std::str::FromStr;
use structopt::clap::AppSettings;
use structopt::StructOpt;

#[derive(Debug)]
pub enum CommandError {
    Unknown,
}

impl fmt::Display for CommandError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            CommandError::Unknown => write!(f, "Unknown flag."),
        }
    }
}

#[derive(StructOpt, Debug, Clone)]
pub enum Command {
    #[structopt(visible_alias = "bac", about = "Show a backlight notification", setting = AppSettings::ColoredHelp)]
    Backlight {
        #[structopt(help = "Specify how long (in seconds) notifications appear for")]
        timeout: Option<u32>,
        /// Specify an icon theme.
        #[structopt(
            short = "I",
            help = "Specify the name of an installed icon theme e.g. \"Adwaita\""
        )]
        icon_theme: Option<String>,
    },
    #[structopt(visible_alias = "bat", about = "Show a battery notification", setting = AppSettings::ColoredHelp)]
    Battery {
        #[structopt(short, help = "Specify how long (in seconds) notifications appear for")]
        timeout: Option<u32>,
        #[structopt(
            short = "I",
            help = "Specify the name of an installed icon theme e.g. \"Adwaita\""
        )]
        icon_theme: Option<String>,
    },
    #[structopt(visible_alias = "n", about = "Show a network notification", setting = AppSettings::ColoredHelp)]
    Network {
        /// Specify an interface for the network notifier.
        #[structopt(short, help = "Specify the interface for the network notification")]
        interface: Option<String>,
        #[structopt(short, help = "Specify how long (in seconds) notifications appear for")]
        timeout: Option<u32>,
        #[structopt(
            short = "I",
            help = "Specify the name of an installed icon theme e.g. \"Adwaita\""
        )]
        icon_theme: Option<String>,
    },
    #[structopt(visible_alias = "u", about = "Show an uptime notification", setting = AppSettings::ColoredHelp)]
    Uptime {
        #[structopt(short, help = "Specify how long (in seconds) notifications appear for")]
        timeout: Option<u32>,
        #[structopt(
            short = "I",
            help = "Specify the name of an installed icon theme e.g. \"Adwaita\""
        )]
        icon_theme: Option<String>,
    },
    #[structopt(visible_alias = "m", about = "Show a memory notification", setting = AppSettings::ColoredHelp)]
    Memory {
        #[structopt(short, help = "Specify how long (in seconds) notifications appear for")]
        timeout: Option<u32>,
        #[structopt(
            short = "I",
            help = "Specify the name of an installed icon theme e.g. \"Adwaita\""
        )]
        icon_theme: Option<String>,
    },
    #[structopt(visible_alias = "d", about = "Show a date notification", setting = AppSettings::ColoredHelp)]
    Date {
        #[structopt(short, help = "Specify how long (in seconds) notifications appear for")]
        timeout: Option<u32>,
        #[structopt(short = "R", long = "railway", help = "Use railway time")]
        railway: bool,
        #[structopt(
            short = "I",
            help = "Specify the name of an installed icon theme e.g. \"Adwaita\""
        )]
        icon_theme: Option<String>,
    },
}

impl fmt::Display for Command {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Command::Backlight {
                timeout: _,
                icon_theme: _,
            } => write!(f, "Backlight"),
            Command::Battery {
                timeout: _,
                icon_theme: _,
            } => write!(f, "Battery"),
            Command::Network {
                interface: _,
                timeout: _,
                icon_theme: _,
            } => write!(f, "Network"),
            Command::Uptime {
                timeout: _,
                icon_theme: _,
            } => write!(f, "Uptime"),
            Command::Memory {
                timeout: _,
                icon_theme: _,
            } => write!(f, "Memory"),
            Command::Date {
                railway: _,
                timeout: _,
                icon_theme: _,
            } => write!(f, "Date"),
        }
    }
}

impl FromStr for Command {
    type Err = CommandError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match &s.to_lowercase()[..] {
            "backlight" => Ok(Command::Backlight {
                timeout: None,
                icon_theme: Some(s.to_string()),
            }),
            "battery" => Ok(Command::Battery {
                timeout: None,
                icon_theme: Some(s.to_string()),
            }),
            "network" => Ok(Command::Network {
                interface: None,
                timeout: None,
                icon_theme: Some(s.to_string()),
            }),
            "uptime" => Ok(Command::Uptime {
                timeout: None,
                icon_theme: Some(s.to_string()),
            }),
            "memory" => Ok(Command::Memory {
                timeout: None,
                icon_theme: Some(s.to_string()),
            }),
            "date" => Ok(Command::Date {
                railway: false,
                timeout: None,
                icon_theme: Some(s.to_string()),
            }),
            _ => Err(CommandError::Unknown),
        }
    }
}
#[derive(Debug, StructOpt, Default)]
#[structopt(settings = &[AppSettings::ColoredHelp, AppSettings::ArgRequiredElseHelp, AppSettings::DisableHelpSubcommand])]
pub struct Options {
    /// Notifies you about your system's uptime.
    #[structopt(subcommand, help = "Send a notification with uptime information")]
    pub cmd: Option<Command>,

    /// Lists the available icon themes.
    #[structopt(short = "L", help = "Returns a list of the available icon themes")]
    pub icon_themes: bool,

    /// Provides debug information.
    #[structopt(long = "debug", short = "D", help = "Provides debug information")]
    pub debug: bool,
}
