use crate::options::Command;
use crate::Icon;
use libmacchina::traits::ReadoutError;
use notify_rust::Notification;
use notify_rust::NotificationHandle;
use notify_rust::Timeout;
use std::fmt;

pub struct Notifier {
    /// The header of a notification.
    summary: Summary,
    /// The body of a notification.
    body: String,
    /// The timeout of a notification.
    timeout: u32,
    /// A path to an icon as a string.
    icon: String,
}

impl Notifier {
    pub fn new() -> Notifier {
        Notifier {
            summary: Summary::Date,
            timeout: 5,
            icon: String::new(),
            body: String::new(),
        }
    }

    pub fn set_summary(&mut self, summary: Summary) {
        self.summary = summary;
    }

    pub fn get_summary(&self) -> String {
        self.summary.to_string()
    }

    pub fn set_icon(&mut self, icon: &Icon) {
        let path = icon.get_path();

        if !path.exists() {
            return;
        }

        if let Some(str) = path.to_str() {
            self.icon = str.to_string();
        }
    }

    pub fn set_timeout(&mut self, timeout: Option<u32>) {
        if let Some(t) = timeout {
            self.timeout = t;
        }
    }

    pub fn get_timeout(&self) -> Timeout {
        Timeout::Milliseconds(self.timeout * 1000)
    }

    pub fn set_body(&mut self, body: Result<String, ReadoutError>) {
        match body {
            Ok(b) => self.body = b,
            Err(e) => self.body = e.to_string(),
        }
    }

    pub fn get_body(&self) -> String {
        self.body.to_owned()
    }

    pub fn send(&self) -> NotificationHandle {
        Notification::new()
            .body(&self.get_body())
            .summary(&self.get_summary())
            .timeout(self.get_timeout())
            .icon(&self.icon)
            .show()
            .expect("Verify that a notification daemon is running in the background")
    }
}

#[derive(Debug, PartialEq, Copy, Clone)]
pub enum Summary {
    Date,
    Uptime,
    Memory,
    Battery,
    Network,
    Backlight,
}

impl fmt::Display for Summary {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Summary::Date => write!(f, "Date"),
            Summary::Uptime => write!(f, "Uptime"),
            Summary::Memory => write!(f, "Memory"),
            Summary::Battery => write!(f, "Battery"),
            Summary::Network => write!(f, "Network"),
            Summary::Backlight => write!(f, "Backlight"),
        }
    }
}

impl From<&Command> for Summary {
    fn from(cmd: &Command) -> Self {
        match cmd {
            Command::Date {
                railway: _,
                timeout: _,
                icon_theme: _,
            } => Summary::Date,
            Command::Uptime {
                timeout: _,
                icon_theme: _,
            } => Summary::Uptime,
            Command::Memory {
                timeout: _,
                icon_theme: _,
            } => Summary::Memory,
            Command::Battery {
                timeout: _,
                icon_theme: _,
            } => Summary::Battery,
            Command::Network {
                interface: _,
                timeout: _,
                icon_theme: _,
            } => Summary::Network,
            Command::Backlight {
                timeout: _,
                icon_theme: _,
            } => Summary::Backlight,
        }
    }
}
