#![allow(unused_must_use)]

// The flow of citron is as follows:
//
//   read   ---->   format   ---->   notify
//    |              |                |
//    |              |                |
//    |              |                `---> `notifier` module
//    |              |
//    |              `---> `fmt` modules
//    |
//    `--> `read` module

mod debug;
mod fmt;
mod helpers;
mod icon;
mod notifier;
mod options;
mod read;

use debug::get_debug;
use icon::Icon;
use notifier::{Notifier, Summary};
use options::Command;
use options::Options;
use owo_colors::OwoColorize;
use structopt::StructOpt;

fn main() {
    let mut notifier = Notifier::new();
    let mut icon = Icon::new();
    let opt = Options::from_args();

    if opt.debug {
        get_debug();
        return;
    }

    if opt.icon_themes {
        println!("Available icon themes:");
        icon.get_available_themes()
            .into_iter()
            .filter(|x| !x.eq("default"))
            .for_each(|x| eprintln!(" - {}", x.yellow()));
    }

    if let Some(cmd) = opt.cmd {
        notifier.set_summary(Summary::from(&cmd));

        match cmd {
            Command::Backlight {
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::backlight(&mut icon));
                notifier.set_timeout(time);
            }
            Command::Battery {
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::battery(&mut icon));
                notifier.set_timeout(time);
            }
            Command::Uptime {
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::uptime(&mut icon));
                notifier.set_timeout(time);
            }
            Command::Network {
                interface: i,
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::network(i.as_deref(), &mut icon));
                notifier.set_timeout(time);
            }
            Command::Date {
                railway: true,
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::date(true, &mut icon));
                notifier.set_timeout(time);
            }
            Command::Date {
                railway: false,
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::date(false, &mut icon));
                notifier.set_timeout(time);
            }
            Command::Memory {
                timeout: time,
                icon_theme: theme,
            } => {
                icon.set_theme(theme);
                notifier.set_body(read::memory(&mut icon));
                notifier.set_timeout(time);
            }
        }

        notifier.set_icon(&icon);
        notifier.send();
    }
}
